(* a few constants *)
let population_size = 20
and max_stagnant_iterations = 1000
and mutation_rate = 0.05;;

(* the type definitions. note the 'and' to string them together in a mutually
   recursive blob so wi and node can refer to eachother *)
type wi = {
	winame: string;
	node: node ref;
	group: group ref;
} and group = {
	essid: string;
	groupwis: wi ref list;
} and node = {
	nodename: string;
	nodewis: wi list;
};;

let nodes = Hashtbl.create 4;;
let groups = Hashtbl.create 4;;

type score = int
 and channel = int;;

let compose f g = fun x -> f(g(x));;
let ($) = compose;;
let maketuple a b = (a, b);;
(* given a hashtable, return all the keys as a list *)
let keys t = Hashtbl.fold (fun k d a -> k::a) t [];;
(* given a hashtable, return all the values as a list *)
let values t = Hashtbl.fold (fun k d a -> d::a) t [];;

(* given a list, return a list of pairs with all possible combinations of 
   items from the given list *)
let rec combinations l =
	match l with
	  []	-> []
	| x::xs	-> (List.map (maketuple x) xs)@(combinations xs)
;;

(* given a configuration and two wi's, return the score *)
let wi_score c wi1 wi2 =
	let scoretable = [ ((>)  2,  1);
			   ((==) 2, -1);
			   ((==) 1, -5);
			   ((==) 0, -10) ] in
	let channel1 = c !(wi1.group) in
	let channel2 = c !(wi2.group) in
	let diff = abs (channel1 - channel2) in
	let rec runtable t = match t with
				[]		-> assert false
			     | (cond, s)::xs	-> if (cond diff) then s
						   else runtable xs in
	runtable scoretable;;

let node_score c n =
	let foldfunc acc (wi1, wi2) = acc + (wi_score c wi1 wi2) in
	List.fold_left foldfunc 0 (combinations n.nodewis);;

let score_configuration c (ns: node list) =
	let mapper g = Hashtbl.find c g.essid in
	let foldfunc acc n = acc + (node_score mapper n) in
	List.fold_left foldfunc 0 ns;;

let add_wi_to_node n (w: wi) = { n with nodewis = (w::n.nodewis) };;

(* given a filename, return a list of all the lines in the file with the given
   filename *)
let snarf_lines fname =
	let infile = open_in fname in
	let result = ref [] in
	try
		while true do
			result := (input_line infile)::!result
		done;
		!result	(* never gets here *)
	with End_of_file -> !result

let parse_file fname =
	let bogus_node = { nodename = "foo"; nodewis = [] } in
	Hashtbl.add nodes fname bogus_node;
	Hashtbl.add groups fname 1;
	;;

let random_configuration =
	let conf = Hashtbl.create 30 in
	Hashtbl.iter (fun k d -> Hashtbl.add conf k (1 + (Random.int 12))) groups;
	conf

let mutate p i = 
	Hashtbl.iter (fun essid chan -> let f = Random.float 1.0 in
					if (f < mutation_rate) then
					  Hashtbl.replace p.(i) essid (1 + (Random.int 13))) p.(i);;

let main = 
	List.iter parse_file (snarf_lines Sys.argv.(1));
	Random.self_init();
	let population = Array.init population_size (fun _ -> random_configuration) in
	let last_high_score = ref (-1000000) in
	let iterations_since_new_high_score = ref 0 in
	let all_nodes = values nodes in
	while !iterations_since_new_high_score < max_stagnant_iterations do
		(* mutate the population *)
		for i = 0 to (population_size / 2 - 1) do
			let i2 = i + population_size / 2 in
			population.(i2) <- Hashtbl.copy population.(i);
			mutate population (i2)
		done;
		(* sort the populations according to score. to do this, make
		   a list of (score, solution) tuples *)
		let score_pop = Array.map (fun c -> ((score_configuration c all_nodes), c)) population in
		(* sort on the first field *)
		Array.sort (fun x y -> compare (fst x) (fst y)) score_pop;
		(* now look at the best score and update the highscore if
		   necessary *)
		let high_score = fst score_pop.(0) in
		if high_score > !last_high_score then begin
			last_high_score := high_score;
			iterations_since_new_high_score := 0
		end;
		(*print_int high_score;*)
		print_int !iterations_since_new_high_score;
		print_newline();
		incr iterations_since_new_high_score
	done;
	()
;;

main
