#!/bin/sh -
# Based on /etc/rc.firewall
#
# Firewall for use at Wireless Leiden Proxy. Rules:
# - Fully exposed internal interface.
# - Allow private networks connected to IFACE_INET to connected to WL NETWORK (NAT).
# - Limited set of outbound NAT for WL connections.
#
# Richard van Mansom <richard@vanmansom.net>
# Rick van der Zwet <info@rickvanderzwet.nl>

# Suck in the configuration variables.
if [ -z "${source_rc_confs_defined}" ]; then
	if [ -r /etc/defaults/rc.conf ]; then
		. /etc/defaults/rc.conf
		source_rc_confs
	elif [ -r /etc/rc.conf ]; then
		. /etc/rc.conf
	fi
fi

IFACE_INET=${IFACE_INET:-sis0}
IFACE_WL=${IFACE_WL:-sis1}
PRIVATE_NETWORKS=192.168.0.0/16,172.16.0.0/12,10.0.0.0/8
# MAX_CONN allowed via Outbound NAT to ONP ports
ONP=http,https
MAX_CONN=10


setup_loopback () {
	############
	# Only in rare cases do you want to change these rules
	#
	${fwcmd} add 100 pass all from any to any via lo0
	${fwcmd} add 200 deny all from any to 127.0.0.0/8
	${fwcmd} add 300 deny ip from 127.0.0.0/8 to any
}

############
# Set quiet mode if requested
#
case ${firewall_quiet} in
[Yy][Ee][Ss])
	fwcmd="/sbin/ipfw -q"
	;;
*)
	fwcmd="/sbin/ipfw"
	;;
esac

############
# Flush out the list before we begin.
#
${fwcmd} -f flush

setup_loopback

${fwcmd} add 400 check-state


############
# Allow all outgoing connections from this system
${fwcmd} add 500 allow all from me to any keep-state

############
# Default services on External Interface (firewalled)
${fwcmd} add 501 allow tcp from any to me ssh via ${IFACE_INET} keep-state
# Default services on Internal Interface (allow all)
${fwcmd} add 502 allow ip from any to me via ${IFACE_WL} keep-state


############
# Inbound NAT setup (External -> WL Network, all ports allowed
# Used to allow external network to use WL Network
${fwcmd} add 1000 nat 1 ip from ${PRIVATE_NETWORKS} to 172.16.0.0/12 out xmit ${IFACE_WL} setup established
${fwcmd} add 1001 nat 1 ip from 172.16.0.0/12 to ${PRIVATE_NETWORKS} in recv ${IFACE_WL} established
${fwcmd} nat 1 config if ${IFACE_WL}
#
${fwcmd} add 1010 allow all from ${PRIVATE_NETWORKS} to 172.16.0.0/12 in via ${IFACE_INET} keep-state

############
# Outbound NAT setup (WL Network -> External, limited portset)
${fwcmd} add 2000 nat 2 ip from 172.16.0.0/12 to any ${ONP} out xmit ${IFACE_INET} setup established
${fwcmd} add 2001 nat 2 ip from any ${ONP} to 172.16.0.0/12 in recv ${IFACE_INET} established
${fwcmd} nat 2 config if ${IFACE_INET}
#
# Block any communication private network at external interface, 
# to ensure you cannot reach the (private network in between). (extra security)
${fwcmd} add 2010 deny all from any to ${PRIVATE_NETWORKS} in recv ${IFACE_WL} 
#
# Allow communication to allowed ports.
${fwcmd} add 2011 allow all from 172.16.0.0/12 to any ${ONP} in via ${IFACE_WL} setup limit src-addr ${MAX_CONN}



############
# Transproxy/WLportal/Captive portal
# ${fwcmd} add 10000 allow tcp from any to localhost 80
# ${fwcmd} add 10001 allow tcp from any to me 80

############
# Reserved: WLPortal rule numbers
# 10010 - 10099

# Forward rules work without a base address, so needed a loop over all inet4 adresses
# for IP in `ifconfig -a | awk '/inet / {print $2}'`; do
#   ${fwcmd} add 10100 fwd $IP,8081 tcp from any to not 172.16.0.0/12 80
# done

############
# By default DENY everything.
${fwcmd} add 65000 deny log logamount 500 all from any to any
