#!/usr/bin/env python
# vim:ts=2:et:sw=2:ai
#
# Build topological network graph
# Rick van der Zwet <info@rickvanderzwet.nl>

import cgi
import gformat
import re
import sys
import urllib
import yaml
import math

def get_yaml(gfile):
  """ Get configuration yaml for 'item'"""
  f = open(gfile, 'r')
  datadump = yaml.load(f)
  return datadump

def write_yaml(gfile, datadump):
  """ Write configuration yaml for 'item'"""
  f = open(gfile, 'w')
  f.write(yaml.dump(datadump, default_flow_style=False))
  f.close()

CACHE_FILE = '/tmp/rd2etrs.yaml'
coordinates = None

def etrs2rd(lam, phi):
  """ Convert rd to etrs """
  
  item = dict()
  (remainder, item['lam_deg']) = math.modf(lam)
  remainder *= 60
  (remainder, item['lam_min']) = math.modf(remainder)
  item['lam_sec'] = remainder * 60

  (remainder, item['phi_deg']) = math.modf(phi)
  remainder *= 60
  (remainder, item['phi_min']) = math.modf(remainder)
  item['phi_sec'] = remainder * 60

  item['func'] = 'etrs2rd'

  args = "&".join(["%s=%s" % (k,v) for k,v in item.iteritems()])
  url = 'http://www.rdnap.nl/cgi-bin/rdetrs.pl?%s' % args
  print "### Fetching coordinate %s, %s using: %s" % (phi, lam, url) 
  f = urllib.urlopen(url)
  raw = f.read()
  
  r = re.compile('name="([a-z_]+)" value="([0-9\.]+)"')
  for i in r.finditer(raw):
    name, value = i.group(1,2)
    value = float(value)
    item[name] = value
  return (item['xrd'], item['yrd'])


def rd2etrs(xrd, yrd, hnap=0.0):
  """ Convert rd to etrs """
  # Get cache is exists
  global coordinates
  if coordinates == None:
    try: 
      coordinates = get_yaml(CACHE_FILE)
    except (IOError,AttributeError):
      coordinates = dict()
      pass

  # Check if item in cache
  xrd = float(str(xrd))
  yrd = float(str(yrd))
  if coordinates.has_key((xrd, yrd)):
    return coordinates[(xrd, yrd)]

  # Get new coordinate
  item = dict()
  item['xrd'] = xrd
  item['yrd'] = yrd
  item['hnap'] = hnap
  item['func'] = 'rd2etrs'
  args = "&".join(["%s=%s" % (k,v) for k,v in item.iteritems()])
  url = 'http://www.rdnap.nl/cgi-bin/rdetrs.pl?%s' % args
  print "### Not in Cache, Fetching coordinate %s, %s from %s" % (xrd, yrd, url) 
  f = urllib.urlopen(url)
  raw = f.read()
  
  r = re.compile('name="([a-z_]+)" value="([0-9\.]+)"')
  for i in r.finditer(raw):
    name, value = i.group(1,2)
    value = float(value)
    item[name] = value
  
  lam = item['lam_deg'] + (item['lam_min'] + (item['lam_sec'] / 60)) / 60
  phi = item['phi_deg'] + (item['phi_min'] + (item['phi_sec'] / 60)) / 60
  coordinates[(xrd, yrd)] = (lam, phi)
  write_yaml(CACHE_FILE, coordinates)
  return (lam, phi)

def make_graph():
  f = open('kmlfile.kml', 'w')
  f.write("""<?xml version="1.0" encoding="UTF-8"?>
<kml xmlns="http://earth.google.com/kml/2.0">
  <Document>
    <name>WirelessLeiden Nodemap</name>
    <open>1</open>
    <description>Generated realtime status of all Wireless Leiden AccessPoints</description>
    <Style id="node_status_up">
      <IconStyle>
        <scale>0.5</scale>
        <Icon><href>http://www.google.com/mapfiles/kml/paddle/grn-stars-lv.png</href></Icon>  
      </IconStyle>
    </Style>
    <Style id="node_status_down">
      <IconStyle>
        <scale>0.5</scale>
        <Icon><href>http://www.google.com/mapfiles/kml/paddle/red-stars-lv.png</href></Icon>  
      </IconStyle>
    </Style>
    <Style id="node_status_planned">
      <IconStyle>
        <scale>0.5</scale>
        <Icon><href>http://www.google.com/mapfiles/kml/paddle/wht-stars-lv.png</href></Icon>  
      </IconStyle>
    </Style>
    <Folder>
      <name>Nodes</name>
      <visibility>0</visibility>
      <description>All active nodes and links</description>
    """)

  poel = {}
  link_type = {}
  node = {}
  
  nodes = []
  links = []
  try:
    for host in gformat.get_hostlist():
      print "## Processing host", host
      datadump = gformat.get_yaml(host)
      iface_keys = [elem for elem in datadump.keys() if (elem.startswith('iface_') and not "lo0" in elem)]
      for iface_key in iface_keys:
        l = datadump[iface_key]['ip']
        addr, mask = l.split('/')

        addr = gformat.parseaddr(addr)
        mask = int(mask)
        addr = addr & ~((1 << (32 - mask)) - 1)
        if poel.has_key(addr): 
          poel[addr] += [host]
        else: 
          poel[addr] = [host]
          # Assume all eth2wifibridge to be 11a for a moment
          iface_parent = '_'.join(iface_key.split('_')[0:2])
          if datadump[iface_parent].has_key('extra_type') and datadump[iface_parent]['extra_type'] == 'eth2wifibridge':
            link_type[addr] = '11a'
          else:
            link_type[addr] = datadump[iface_parent]['type']
          print "### %s [%s] is of type %s" % (gformat.showaddr(addr), iface_key, link_type[addr])
      lam, phi = rd2etrs(datadump['rdnap_x'], datadump['rdnap_y'])
      node[host] = (lam, phi)
      f.write("""
      <description>All active nodes</description>
      <Placemark>
        <name>Node %(name)s</name>
        <description>%(desc)s</description>
        <styleUrl>#node_status_up</styleUrl>
        <Point><coordinates>%(lam)s,%(phi)s,0</coordinates></Point>
      </Placemark>
   """ % {'name' : host, 'desc' : cgi.escape(datadump['location']), 'lam' : lam, 'phi' : phi})
      nodes += [("POINT(%s, %s)" % (lam, phi))]
  except (KeyError, ValueError), e:
    print "[FOUT] in '%s' interface '%s'" % (host,iface_key) 
    raise
    sys.exit(1)

  f.write("""
    </Folder>
    <Folder>
      <name>Links</name>
      <visibility>0</visibility>
      <description>All links</description>
  """)
  for addr,leden in poel.iteritems():
    if link_type[addr] == '11a':
      color = '#ff0000ff'
      weight = 2
    elif link_type[addr] == 'eth':
      color = '#ffff0000'
      weight = 4
    else:
      color = '#ff000000'
      weight = 1

    leden = sorted(set(leden))
    for index,lid in enumerate(leden[:-1]):
      for buur in leden[index + 1:]:
        f.write("""
      <Placemark>
        <name>%(name)s</name>
        <visibility>0</visibility>
        <description>%(desc)s</description>
        <LineString>
          <tessellate>0</tessellate>
          <coordinates> %(lam1)s, %(phi1)s, 0
            %(lam2)s , %(phi2)s, 0 </coordinates>
        </LineString>
        <Style>%(style)s</Style>
      </Placemark>
      """ % { 'lam1' : node[lid][0], 
              'phi1' : node[lid][1], 
              'lam2' : node[buur][0], 
              'phi2' : node[buur][1],
              'name' : "Interlink: %s --- %s" % (lid, buur), 
              'desc' : "%s [%s]" % (gformat.showaddr(addr), link_type[addr]),
              'style' : "<LineStyle><color>%s</color><width>%s</width></LineStyle>" % (color, weight),
              })
  f.write("""
    </Folder>
  </Document>
</kml>
  """)
  f.close()


if __name__ == "__main__":
  make_graph()

