#!/usr/bin/env python
#
# Wrap me around tcpserver or inetd, example usage for tcpserver (debug):
# tcpserver -HRl localhost 172.31.255.1 /root/wlportal.py
#
# Or put me in a CGI script in for example thttpd server:
#
# = Usage =
# This is a wrapper script which does very basic HTML parsing and altering of
# ipfw tables rules to build a basic Captive Portal, with basic sanity
# checking. The ACL is IP based (this is a poor mans solution, layer2
# ACL would be much better), so don't take security very seriously.
# 
# To get traffic by default to the portal iI requires a few special rules in
# ipfw to work properly (ajust IP details if needed):
# - Rule 10010-10099 needs to be free.
# - add 10100 fwd 172.20.145.1,8081 tcp from any to not 172.16.0.0/12 dst-port 80 in via wlan0
# 
# Enties older than 5 minutes not being used will be removed if the (hidden)
# argument action=cleanup is given as GET variable. So having this in cron (would fix it):
# */5 * * * * /usr/bin/fetch -q http://172.31.255.1/wlportal?action=cleanup
#
# XXX: The whitelist entries first needs to contact the wlportal.py to get
# added to the whitelist, this may cause issues during initial setup and hence
# it might be advised to create a block of static whitelist IP addresses which
# get added during boot and will never disappear.
#
# The program has uses a file based persistent cache to save authenticated
# ACLs, this will NOT get synced after a reboot. 
#
# State   : ALPHA 
# Version : $Id: index.cgi 10131 2012-03-12 03:33:57Z richardvm $
# Author  : Rick van der Zwet <info@rickvanderzwet.nl>
# Licence : BSDLike http://wirelessleiden.nl/LICENSE

import logging
import os
import pickle
import re
import signal
import subprocess
import sys
import time
import traceback
import urlparse
import yaml

# XXX: Make me dynamic for example put me in the conf file
conf = { 
  'autologin'     : False,
  'cmd_arp'       : '/usr/sbin/arp',
  'cmd_fw'        : '/sbin/ipfw',
  'portal_sponsor': 'Sponsor van Stichting Wireless Leiden',
  'portal_url'    : 'http://www.wirelessleiden.nl',
  'portalroot'    : '172.31.255.1',
  'refresh_delay' : 5,
  'tmpl_autologin': '/usr/local/etc/wlportal/autologin.tmpl',
  'tmpl_login'    : '/usr/local/etc/wlportal/login.tmpl',
  'whitelist'     : [],
}


logging.basicConfig(stream=open('/var/log/wlportal.log','a'),level=logging.DEBUG)

# No failback if config does not exist, to really make sure the user knows if
# the config file failed to parse properly or is non-existing
# XXX: 5xx error code perhaps?
try:
  conf.update(yaml.load(open('/usr/local/etc/wlportal/config.yaml')))
except Exception,e:
  logging.error(traceback.format_exc())


class ItemCache:
  """ 
  Very basic ItemCache used for caching registered entries and other foo, no
  way recurrent, so use with care!
  """

  def __init__(self, authentication_timeout=60):
    self.cachefile='/tmp/portal.cache'
    # cache[mac_address] = (ipaddr, registered_at, last_seen)
    self.cache = None
    self.arp_cache = None
    self.now = time.time()
    self.authentication_timeout = authentication_timeout

  def delete_all(self):
    self.cache = {}
    self.save()

  def delete(self,ipaddr):
    self.load()
    for mac in self.cache.keys():
      if self.cache[mac][0] == ipaddr:
        del self.cache[mac]
    self.save()


  def load(self):
    """ Request cached file entries """
    if self.cache == None:
      try:
        self.cache = pickle.load(open(self.cachefile,'r'))
      except IOError:
        self.cache = {}
        pass

  def load_arp_cache(self):
    """ Provide with listing of MAC to IP numbers """
    if self.arp_cache == None:
       output = subprocess.Popen([conf['cmd_arp'],'-na'], stdout=subprocess.PIPE).communicate()[0]
       self.arp_cache = {}
       for line in output.strip().split('\n'):
         # ? (172.20.145.30) at 00:21:e9:e2:7c:c6 on wlan0 expires in 605 seconds [ethernet]
         if not 'expires' in line:
           continue
         t = re.split('[ ()]',line)
         ip, mac = t[2],t[5]
         self.arp_cache[ip] = mac 

  def get_mac(self,ipaddr):
    self.load_arp_cache()
    try:
      return self.arp_cache[ipaddr]
    except KeyError:
      return None

  def add(self,ipaddr):
    """ Add entry to cache (on file) and return entry"""
    self.load()
    self.load_arp_cache()
    self.cache[self.arp_cache[ipaddr]] = (ipaddr, self.now, self.now)
    logging.debug("Adding Entry to Cache %s -> %s" % (ipaddr, self.arp_cache[ipaddr]))
    self.save()

  def save(self):
    """ Sync entries to disk """
    # XXX: Should actually check if entry has changed at all
    pickle.dump(self.cache, open(self.cachefile,'w'))

  def update():
    """ Update entries with relevant ARP cache """
    self.load()
    self.load_arp_cache()
    # Update last_seen time for currently active entries
    for ip,mac in self.arp_cache.iteritems():
      if self.cache.has_key(mac):
        self.cache[mac][3] = now
    
    # cleanup no longer used entries, after authentication_timeout seconds.
    for mac in self.cache:
      if self.cache[mac][3] < self.now - self.authentication_timeout:
        del self.cache[mac]

    # Sync results to disk
    self.save()
    return self.cache

  def get_cache(self):
    self.load()
    return self.cache

  def get_arp_cache(self):
    self.load_arp_cache()
    return self.arp_cache


class FirewallControl:
  def __init__(self):
    self.first_rule = 10010
    self.last_rule  = 10099
    self.available_rule = self.first_rule
    self.logger = ''
  

  def load(self):
    # Get all registered ips
    sp =  subprocess.Popen([conf['cmd_fw'],'show','%i-%i' % (self.first_rule, self.last_rule)], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    output = sp.communicate()[0]
    self.ip_in_firewall = {}
    if sp.returncode == 0:
      # 10010   32   1920 allow tcp from 172.20.145.30 to not 172.16.0.0/12 dst-port 80
      for line in output.strip().split('\n'):
        t = line.split()
        rule, ip = t[0], t[6] 
        self.ip_in_firewall[ip] = rule
        
        if self.available_rule == int(rule):
          self.available_rule += 1
    else:
      # XXX: Some nagging about no rules beeing found perhaps?
      pass

  def cleanup(self):
    """ Cleanup Old Entries, mostly used for maintenance runs """
    self.load()
    # Make sure cache matches the latest ARP version
    itemdb = ItemCache()
    cache = itemdb.get_cache()
    valid_ip = itemdb.get_arp_cache()
      
    # Check if all ipfw allowed entries still have the same registered MAC address
    # else assume different user and delete. 
    for ip,rule in self.ip_in_firewall.iteritems():
      delete_entry = False
    
      # Make sure IP is still valid
      if not valid_ip.has_key(ip):
        delete_entry = True
      # Also MAC needs to exists in Cache
      elif not cache.has_key(valid_ip[ip]):
        delete_entry = True
      # IP need to match up with registered one
      elif not cache[valid_ip[ip]][0] == ip:
        delete_entry = True
    
      # Delete entry if needed
      if delete_entry:
        output = subprocess.Popen([conf['cmd_fw'],'delete',str(rule)], stdout=subprocess.PIPE).communicate()[0]
        self.logger += "Deleting ipfw entry %s %s\n" % (rule, ip)
        logging.debug('Deleting ipfw entry %s %s\n' % (rule, ip))


  def add(self,ipaddr):
    """ Add Entry to Firewall, False if already exists """
    self.load()
    if not self.ip_in_firewall.has_key(ipaddr):
      rule = "NUMBER allow tcp from IPADDR to not 172.16.0.0/12 dst-port 80".split()
      rule[0] = str(self.available_rule)
      rule[4] = str(ipaddr)
      logging.debug("Addding %s" % " ".join(rule))
      output = subprocess.Popen([conf['cmd_fw'],'add'] + rule, stdout=subprocess.PIPE).communicate()[0]
      itemdb = ItemCache()
      itemdb.add(ipaddr)
      self.register(ipaddr)
      return True
    else:
      return False

  def register(self, ipaddr):
    epoch = int(time.time())

    itemdb = ItemCache()
    mac = itemdb.get_mac(ipaddr)

    filename = "/var/db/clients" 
    file = open(filename,"a")
    file.write("%s %s %s \n" % (epoch, mac, ipaddr) )
    file.close()


  def delete(self, ipaddr):
    itemdb = ItemCache()
    itemdb.delete(ipaddr)
    self.cleanup()

  def delete_all(self):
    itemdb = ItemCache()
    itemdb.delete_all()
    self.cleanup()
 
  def get_log(self):
    return self.logger



# Query String Dictionaries
qs_post = None
qs = None
header = []

# Hybrid Setup.
# a) We are not wrapped around in a HTTP server, so this _is_ the
#    HTTP server, so act like one.
if not os.environ.has_key('REQUEST_METHOD'):
  class TimeoutException(Exception):
    """ Helper for alarm signal handling"""
    pass
  
  def handler(signum, frame):
    """ Helper for alarm signal handling"""
    raise TimeoutException
  
  
  # Parse the HTTP/1.1 Content-Header (partially)
  signal.signal(signal.SIGALRM,handler)
  us = None
  method = None
  hostname = None
  content_length = None
  remote_host = None
  while True:
    try:
      signal.alarm(1)
      line = sys.stdin.readline().strip()
      if not line:
        break
      header.append(line)
      signal.alarm(0)
      if line.startswith('GET '):
        us = urlparse.urlsplit(line.split()[1])
        method = 'GET'
      elif line.startswith('POST '):
        method = 'POST'
        us = urlparse.urlsplit(line.split()[1])
      elif line.startswith('Host: '):
        hostname = line.split()[1]
      elif line.startswith('Content-Length: '):
        content_length = int(line.split()[1])
    except TimeoutException:
      break
  
  # Capture Portal, make sure to redirect all to portal
  if hostname != conf['portalroot']:
    print "HTTP/1.1 302 Moved Temponary\r\n",
    print "Location: http://%(portalroot)s/\r\n" % conf,
    sys.exit(0)
  
  
  # Handle potential POST
  if method == 'POST' and content_length:
    body = sys.stdin.read(content_length)
    qs_post = urlparse.parse_qs(body)
  
  # Parse Query String
  if us and us.path == "/wlportal" and us.query:
    qs = urlparse.parse_qs(us.query)

  remote_host = os.environ['REMOTEHOST']
else:
  # b) CGI Script: Parse the CGI Variables if present
  if os.environ['REQUEST_METHOD'] == "POST":
    content_length = int(os.environ['CONTENT_LENGTH'])
    body = sys.stdin.read(content_length)
    qs_post = urlparse.parse_qs(body)

  if os.environ.has_key('QUERY_STRING'):
    qs = urlparse.parse_qs(os.environ['QUERY_STRING'])

  remote_host = os.environ['REMOTE_ADDR']


# Helpers for HTML 'templates'
content = conf.copy()
content.update(extra_header='',tech_footer='',status_msg='')

# IP or MAC on the whitelist does not need to authenticate, used for devices
# which need to connect to the internet, but has no 'buttons' to press OK.
#
# This assumes that devices will re-connect if they are not able to connect 
# to their original host, as we do not preserve the original URI.
ic = ItemCache()
if conf['autologin'] or remote_host in conf['whitelist'] or ic.get_mac(remote_host) in conf['whitelist']:
  qs_post = { 'action' : 'login' }

try:
  # Put authenticate use and process response
  if qs and qs.has_key('action'):
    if 'deleteall' in qs['action']:
      content['status_msg'] += "# [INFO] Deleting all entries\n"
      fw = FirewallControl()
      fw.delete_all()
      content['status_msg'] += fw.get_log()
    elif 'update' in qs['action']:
      tech_footer = "# [INFO] Update timestamp of all entries\n"
      fw = FirewallControl()
      fw.update()
      content['status_msg'] += fw.get_log()
    elif 'cleanup' in qs['action']:
      content['status_msg'] += "# [INFO] Deleting all entries"
      fw = FirewallControl()
      fw.delete_all()
  elif qs_post and qs_post.has_key('action'):
    if 'login' in qs_post['action']:
      fw = FirewallControl()
      if fw.add(remote_host):
        content['extra_header'] = "Refresh: %(refresh_delay)s; url=%(portal_url)s\r" % content
        content['status_msg'] = "Sucessfully Logged In! || " +\
        """ Will redirect you in %(refresh_delay)s seconds to <a href="%(portal_url)s">%(portal_url)s</a> """ % content
      else:
        content['status_msg'] = "ERROR! Already Logged On"
    elif 'logout' in qs_post['action']:
      fw = FirewallControl()
      fw.delete(remote_host)
      content['status_msg'] = "Succesfully logged out!"

except Exception,e:
  content['tech_footer'] += traceback.format_exc()
  content['status_msg'] = e
  pass

  # Present Main Screen
print """\
HTTP/1.1 200 OK\r
Content-Type: text/html\r
%(extra_header)s
""" % content

try:
  tmpl_file = conf['tmpl_autologin'] if conf['autologin'] else conf['tmpl_login']
  page = open(tmpl_file,'r').read()
except IOError:
  page = """
<html><head></head><body>
<h2>%(status_msg)s</h2>

<h3>Wireless Leiden - Internet Portal</h3>
<form action="http://%(portalroot)s/wlportal/" method="POST">
<input name="action" type="hidden" value="login" />
<input type="submit" value="OK, agreed" />
</form>

<h3>More options</h3>
<form action="http://%(portalroot)s/wlportal/" method="POST">
<input name="action" type="hidden" value="logout" />
<input type="submit" value="Cancel and/or Logout" />
</form>
<hr /><em>Technical Details:</em><pre>
%(tech_footer)s
</pre>
</body></html>
"""

print page % content
