#!/bin/sh
# Get nanobsd image ready to be booted from NFS
# NFS instructions at
# http://www.wirelessleiden.nl/projects/nodefactory/wiki/TestingViaNFS

IMAGE_BASE="/usr/obj/nanobsd.wleiden"
IMAGE_SLICE="${IMAGE_BASE}/_.disk.image"
IMAGE_FULL="${IMAGE_BASE}/_.disk.full"
IMAGE_NFS="${IMAGE_SLICE}-nfs"
# Structure: $NFSBASE 
#	     ./cfg  = /cfg mounpoint
#            ./base = / mountpoint
#            ./nfs  = /nfs mountpoint
NFSBASE='/usr/data'
MNT="${NFSBASE}/base"
CFG="${NFSBASE}/cfg"
NFS="${NFSBASE}/nfs"

p_err() {
	echo "[ERROR] $*" 1>&2
}

usage() {
	(
	echo "Usage: $0 [-fn]"
	echo "	-f	force umount, memory device whipes"
	echo "	-n	do not delete/clean cfg partition"
	echo "	-u	unload/eject procedure"
	) 1>&2
	exit 2
}

# No Root, no fun
if [ `id -u` -ne 0 ]; then
	p_err "Root only"
	exit 1
fi

# Argument parsing using getopts
OPT_FORCE=0
OPT_CLEAN=1
OPT_UNLOAD=0
while getopts "hfnu" OPT; do
	case "$OPT" in
	f) OPT_FORCE=1;;
	n) OPT_CLEAN=0;;
	u) OPT_UNLOAD=1;;
	h) usage;;
	\?) usage;;
	esac
done


# Eeks, we are going to be nasty, hold your horses
if [ $OPT_FORCE -eq 1 -o $OPT_UNLOAD -eq 1 ]; then
	umount -f $MNT	
	for MD in `mdconfig -l -v | grep "${IMAGE_NFS}" | awk '{print $1}'`; do
		mdconfig -d -u $MD
	done
fi

if [ $OPT_UNLOAD -eq 1 ]; then
	echo "All done"
	exit 1;
fi

# If mount point is already used, bail out
if mount | grep -q "${MNT}"; then
	p_err "'${MNT}' already mounted"
	exit 1
fi

# If target nfs image is mounted somehow, bail out
if mdconfig -l -v | grep -q "${IMAGE_NFS}"; then
	MD=`mdconfig -l -v | grep "${IMAGE_NFS}" | cut -c -4`
	p_err "'${IMAGE_NFS}' already mounted at '$MD'"
	exit 1
fi


# Prepare image for use with NFS
cp -v ${IMAGE_SLICE} ${IMAGE_NFS}

MD=`mdconfig -a -t vnode -f ${IMAGE_NFS}`
mount /dev/${MD}a ${MNT}

# Config files lives at NFS location
echo "mount -t nfs -o ro 192.168.4.1:${CFG}" > ${MNT}/conf/default/etc/remount

# Create nfs mount location
mkdir ${MNT}/nfs

# $MNT, $CFG, $NFS lives at nfs
(
echo "192.168.4.1:${MNT} / nfs ro 0 0" 
echo "192.168.4.1:${CFG} /cfg nfs rw,noauto 0 0"
echo "192.168.4.1:${NFS} /nfs nfs rw 0 0"
) > /${MNT}/conf/base/etc/fstab

if [ ${OPT_CLEAN} -eq 1 ]; then
	echo "DELETING all files at ${CFG}, start fresh ;-)" 
	rm -vfR ${CFG}/*
else
	echo "PRESERVING all files at ${CFG}"
fi

echo "DELETING all files at ${NFS}, start fresh ;-)" 
rm -vfR ${NFS}/*

# Allow build images to be used directy via NFS
ln -f ${IMAGE_SLICE} ${NFS}/`basename ${IMAGE_SLICE}`
ln -f ${IMAGE_FULL} ${NFS}/`basename ${IMAGE_FULL}`

# XXX: Proper unmounting after all has finished
