#(c) Roland van Laar 2006
# vi:et:ts=4
from django.db import models
from django.db.models import Q

#No need to formalize it; CHOICES don't change much

STATUS_CHOICES = ( 
         ('up', 'up'),
         ('dw', 'down'),
         ('pl', 'planned'),
         )

POLAR_CHOICES = (
         ('hr', 'horizontal'),
         ('vr', 'vertical'),
         )

INTERFACE_TYPE_CHOICES = (
         ('eth', 'eth'),
         ('11a', '11a'),
         ('11b', '11b'),
         ('11g', '11g'),
         )
WIFI_MODE_CHOICES = (
         ('ms', 'master'),
         ('mn', 'managed'),
         )

DIRECTION_CHOICES = (
        ('n', 'north'),
        ('nw', 'north-west'),
        ('w', 'west'),
        ('sw', 'south-west'),
        ('s', 'south'),
        ('se', 'south-east'),
        ('e', 'east'),
        ('ne', 'north-east'),
        ('omni', 'omnidirectional'),
        )


class Antenna(models.Model):
    type = models.CharField(max_length=20, unique=True)
    gain = models.IntegerField(max_length=3)

    class Meta:
        verbose_name = 'Antenna'

    def __unicode__(self):
        return self.type

class EthernetToWifiBridge(models.Model):
    type = models.CharField(max_length=10, unique=True)

    def __unicode__(self):
        return self.type

class Location(models.Model):
    description = models.CharField(max_length=200, unique=True)
    latitude = models.DecimalField(max_digits=16,decimal_places=14)
    longitude = models.DecimalField(max_digits=16,decimal_places=14)

    class Meta:
        verbose_name = 'Location'

    def __unicode__(self):
        return self.description

    def delete(self):
        for i in self.node_set.all():
            i.delete()
        super(Location, self).delete()


#XXX: Name Domain perhaps?
class DnsServer(models.Model):
    ipaddress = models.IPAddressField()
    domainname = models.CharField(max_length=50, unique=True)
    
    class Meta:
        verbose_name = 'Dns Server'
    
    def __unicode__(self):
        return "%s, %s"% (self.domainname, self.ipaddress )

class Network(models.Model):
    name = models.CharField(max_length=50, unique=True)
    ipspacestart = models.IPAddressField()
    netmask = models.IntegerField()

    class Meta:
     verbose_name = 'Network'

    def __unicode__(self):
        return self.name

class Node(models.Model):
    name = models.CharField(max_length=30, unique=True)
    location = models.ForeignKey(Location)
    status = models.CharField(max_length=10, choices=STATUS_CHOICES , \
        default='up')
    network = models.ForeignKey(Network,default=1)
    masterip = models.IPAddressField(unique=True)
 
    class Meta:
        ordering = ['name']
        verbose_name = 'Node'

    def __unicode__(self):
        return self.name

    def delete(self):
        for i in self.interface_set.all():
            i.delete
        super(Node, self).delete()

class Interface(models.Model):

    def all(self):
        return Interface.objects.all()
    node = models.ForeignKey(Node)
    ethernet_to_wifi = models.ForeignKey(EthernetToWifiBridge, blank=True, 
            help_text="Don't forget to add the brigde as a static host",
            null=True)
    type = models.CharField(max_length=10, choices=INTERFACE_TYPE_CHOICES, 
            default='eth')
    iface = models.CharField(max_length=10, verbose_name="interface", 
            default='eth0')
    accesspoint = models.BooleanField()
    polar = models.CharField(blank=True, max_length=10, choices=POLAR_CHOICES)
    antenna = models.ForeignKey(Antenna, blank=True, null=True)
    direction = models.CharField(max_length=10, blank=True, null=True, 
            choices=DIRECTION_CHOICES)
    # default='self' only works on the second save when link is not defined.
    link = models.ForeignKey('self', blank=True, null=True, default='self')
    ip = models.IPAddressField(unique=True, blank=True)
    netmask = models.IntegerField(null=True, blank=True)
    ssid = models.CharField(max_length=50, blank=True, null=True)
    mode = models.CharField(max_length=10, choices=WIFI_MODE_CHOICES,
            blank=True, null=True)
    channel = models.IntegerField(blank=True, null=True)
   
    class Meta:
        unique_together = ('node', 'iface')
        verbose_name = 'Interface'
        #order_with_respect_to = 'node'
        ordering = ['node', 'iface']
   
    def __unicode__(self):
        if self.ethernet_to_wifi:
            return "%s/%s/B" % (self.node, self.iface)
        else:
            return "%s/%s" % (self.node, self.iface)

    def save(self, force_insert=False, force_update=False):
        super(Interface, self).save(force_insert, force_update)
        # If link is not defined we link to self.
        if self.link == None:
            self.link = self
            super(Interface, self).save(False, True)


    def delete(self):
        #import pdb; pdb.set_trace() ;
        # remove reference to all other interfaces.
        self.interface_set.clear()
        super(Interface, self).delete()
        

class DhcpStatic(models.Model):
    hostname = models.CharField(max_length=10,unique=True)
    macaddress = models.CharField(max_length=17)
    accesspoint = models.ForeignKey(Interface, 
            limit_choices_to = Q(accesspoint=True)|Q(ethernet_to_wifi=True) ) 
    ipaddress = models.IPAddressField(blank=True)

    class Meta:
        verbose_name = "Static host"

    def __unicode__(self):
        return "%s.%s.%s %s" % (self.hostname, self.accesspoint.node.name, 
                self.accesspoint.node.network.name, self.macaddress)
