from django import forms
from exodus.models import Location, Node, Interface, DhcpStatic
from exodus.wllogic import free_master_ip, link_has_compat_type, \
        link_is_wireless, new_ssid, new_ssid_for_existing_interface
from exodus.wllogic import MASTER, MANAGED, calc_subnet, network, show_addr
from exodus.settings import AP_NETMASK
from exodus.wlipcalc import IPCalc
import re

class LocationForm(forms.ModelForm):
    class Meta:
        model = Location

class NodeForm(forms.ModelForm):
    class Meta:
       model = Node

    def clean_name(self):
        name = self.cleaned_data.get('name')
        re_name = re.compile('[0-9a-zA-Z-]+')
        if not name == re_name.match(name).group():
            raise forms.ValidationError(
                    'Not a valid name. Use letters, digits and -.')
        return name

    def clean_masterip(self):
        new_network = self.cleaned_data.get('network')
        # self.instnace.pk is only available when Node has been save before.
        if self.instance.pk is None:
            masterip = free_master_ip(new_network)
        else:
            # check if network has changed
            if self.instance.network == new_network:
                masterip = self.cleaned_data.get('masterip')
            else:
                masterip = free_master_ip(new_network) 
                masterlinks = ( i for i in self.instance.interface_set.all()
                        if i.id == i.link_id )
                for i in masterlinks:
                    all_links = i.interface_set.all()
                    if i.accesspoint:
                        netmask = AP_NETMASK
                    else:
                        ip_size = len(all_links)
                        netmask = calc_subnet(ip_size)

                    try:
                        new_ip = IPCalc(self.instance, netmask, i.id, 
                                masterip = masterip)
                    except Exception:
                        raise forms.ValidationError(
                            "Not enough free ips for this node.")

                    new_ip.ips.reverse()
                    i.ip = new_ip.ips.pop()
                    i.ssid = new_ssid(new_network, self.instance, i.iface, 
                            i.accesspoint, i.direction)
                    for j in (j for j in all_links if not j.id == i.pk):
                        j.ip  = new_ip.ips.pop()
                        j.netmask = netmask
                        j.ssid = i.ssid
                        j.save() 
                    i.save()
                    # update the static hosts on this interface
                    for j in i.dhcpstatic_set.all(): 
                        ip_size = int(pow(2,32-netmask))
                        networkaddrL = network(i.ip, i.netmask)
                        # 1, -1 compensate for network and broadcast addr. 
                        iplist = [show_addr(networkaddrL + k) for k in 
                                xrange(1, ip_size-1)]
                        import pdb; pdb.set_trace() ;
                        for k in i.interface_set.all():
                            iplist.remove(k.ip)
                        #remove used dhcpstatic ips, beware that interfaces
                        #that haven't been updated yet have ipaddress that 
                        #are not in iplist.
                        for k in i.dhcpstatic_set.all():
                            try:
                                iplist.remove(k.ipaddress)
                            except ValueError:
                                pass
                        iplist.reverse()
                        j.ipaddress = iplist.pop()
                        j.save()
        return masterip        

class InterfaceForm(forms.ModelForm):
    class Meta:
        model = Interface

    def clean_iface(self):
        iface = self.cleaned_data.get('iface')
        re_iface = re.compile('[a-z]+[0-9]+')
        if not re_iface.match(iface) or \
                re_iface.match(iface).group() != iface:
            raise forms.ValidationError("Please enter a valid interface name")
        return iface

    def clean_polar(self):
        polar = self.cleaned_data.get('polar')
        self.type = self.cleaned_data.get('type')
        
        if polar and not link_is_wireless(self):
            raise forms.ValidationError( 
                    "An ethernet interface can't have a polarization.")
        else:
            return polar

    def clean_antenna(self):
        self.type = self.cleaned_data.get('type')
        antenna = self.cleaned_data.get('antenna')
        if antenna and not link_is_wireless(self):
            raise forms.ValidationError(
                    "An ethernet interface can't have an antenna.")
        return antenna

    def clean_direction(self):
        self.type = self.cleaned_data.get('type')
        direction = self.cleaned_data.get('direction')
        if direction and not link_is_wireless(self):
            raise forms.ValidationError(
                    "An ethernet interface can't have a direction.")
        return direction
        

    def clean_link(self):
        link = self.cleaned_data.get('link')
        # the type of the interface, eth, 11a, 11b, 11g.
        type = self.cleaned_data.get('type')

        # self.instance raises DoesNotExist, 
        # but self.instance.[pk, ssid, polar, etc] doesn't
        edit = bool(self.instance.pk)

        if link:
            # if link is to self we don't need to check anything else
            if self.instance.pk == link.pk:
                return link

            # check if the two links are compatible
            elif not link_has_compat_type(link.type, type):
                raise forms.ValidationError( 
                    'Link type %s is not compatible with %s' 
                    %(link.type, type))

            # Link can't be to same node, unless link is to self, 
            # which we checked above.
            elif self.instance.node_id == link.node_id:
                raise forms.ValidationError(
                    "A link can't be to another interface on the same node")

            elif self.cleaned_data.get('accesspoint') and \
                    self.instance.pk != link.pk:
                raise forms.ValidationError( "A link can't be made to another"
                        " interface when this interface has an accesspoint.")

            # if link is referenced to itself, link is master and linkable
            elif link.link.pk == link.pk:
                return link

            # if link is ethernet, don't worry about master and managed.
            # but we do want every link to link to one 'master' in the
            # subnet.
            # We assume that the link of a link is always the master.
            # if link is to self, link.link_id == link.id
            elif not link_is_wireless(link):
                if link.id == link.link_id:
                    return link
                else:
                    return link.link

            # if this elif is True, it means that 'link' is in managed 
            # mode but the link of link can change modes (master/managed).
            # And check if that master has only 2, including himself, links
            # and is not an accesspoint himself.
            # We don't update the netmasks because 2 links should
            # already have a /30, and the number of links doesn't change
            # here.
            elif len(link.link.interface_set.all()) == 2 and not \
                    link.link.accesspoint:

                # Define the new master and the new slave
                new_master = link
                new_slave = new_master.link
                # set both links to the new master
                new_master.link = new_master
                new_slave.link = new_master
                #update mode(manage/master) when an interface is wireless
                if not link_is_wireless(new_master):
                    new_master.type = MASTER
                    new_slave.type = MANAGED 
                # update ssids
                ssid = new_ssid_for_existing_interface(new_master)
                new_master.ssid = ssid
                new_slave.ssid = ssid

                # calc the new ipaddresses
                ip = IPCalc(new_master.node, 30)  
                new_master.ip, new_slave.ip = ip.ips
                new_master.netmask = new_slave.netmask = 30
                    
                # Save 
                new_master.save()
                new_slave.save()
                return link

            elif len(link.link.interface_set.all()) > 2 or \
                    link.link.accesspoint:
                raise forms.ValidationError( "The other interface is in"
                        " managed mode, and can't be changed to master.") 
            # We shouldn't come here, because all link possibilities are 
            # accounted for.
            else:
                raise forms.ValidationError(
                    "This error should never come up, please report this bug.")
        # If this is an edit and link is not defined, we let link reference 
        # to itself.
        elif edit:
            link = Interface.objects.get(pk=self.instance.pk)
            return link
        # This is a new interface, can't be linked to itself because it hasn't 
        # been saved.
        else:
            return link

    def clean_ip(self):
        """Cleans the ip and netmask.

        We calculate the ipaddresses of this link and the ones linked to it.
        We also define the netmask in this routine.
        """
        node = self.cleaned_data.get('node')
        ip = self.cleaned_data.get('ip')
        link = self.cleaned_data.get('link')   
        ap = self.cleaned_data.get('accesspoint')
        edit = bool(self.instance.pk)

        # if we have an error, we don't calculate new ip addresses.
        if self._errors:
            raise forms.ValidationError(
                "Please correct the errors in the form.")

        if not link:
            if not ap:
                try:
                    new_ip = IPCalc(node, 32)
                except Exception:
                    raise forms.ValidationError(
                        "Not enough free ips for this node.")
                return new_ip.ips[0]

            if ap:
                try:
                    new_ip = IPCalc(node, AP_NETMASK)
                except Exception:
                    raise forms.ValidationError(
                        "Not enough free ips for this node.")
                return new_ip.ips[0]

        else:
            pk = self.instance.pk
            # link is managed
            if link.id != pk:
                all_links = link.interface_set.all()
                ip_size = len(all_links)
                # up ip_size with one if self is not in all_links
                if not [ i for i in all_links if i.id == pk ]:
                    ip_size += 1 
                if link.accesspoint:
                    netmask = AP_NETMASK
                else:
                    netmask = calc_subnet(ip_size)

                try:
                    new_ip = IPCalc(link.node, netmask, link.id)
                except Exception:
                    raise forms.ValidationError(
                        "Not enough free ips for this node.")

                new_ip.ips.reverse()
                link.ip = new_ip.ips.pop()
                link.netmask = netmask
                link.save()
                ip = new_ip.ips.pop()
                # filter out own and master interface
                for i in (i for i in all_links if not i.id in (pk, link.id)):
                    i.ip = new_ip.ips.pop()
                    i.netmask = netmask
                    i.save()
                return ip

            # link is master 
            if link.id == self.instance.pk:
                all_links = link.interface_set.all()
                if ap:
                    netmask = AP_NETMASK
                else:
                    ip_size = len(all_links)
                    # up ip_size with one if self is not in all_links
                    if not [ x for x in all_links if x.id == pk ]:
                        ip_size += 1 
                    netmask = calc_subnet(ip_size)

                try:
                    new_ip = IPCalc(link.node, netmask, link.id)
                except Exception:
                    raise forms.ValidationError(
                        "Not enough free ips for this node.")
                new_ip.ips.reverse()
                ip = new_ip.ips.pop()
                for i in (i for i in all_links if not i.id == pk):
                    i.ip  = new_ip.ips.pop()
                    i.netmask = netmask
                    i.save() 
                return ip

    def clean_netmask(self):
        """Cleaning happens in clean_ip."""
        netmask = self.cleaned_data.get('netmask')
        link = self.cleaned_data.get('link')
        edit = bool(self.instance.pk)
        accesspoint = self.cleaned_data.get('accesspoint')

        if accesspoint:
            return AP_NETMASK

        if not edit and not link:
            return 32
         
        if not edit and link:
            return link.netmask

        if self.instance.pk != link.id:
            return link.netmask
            
        if self.instance.pk == link.id:
            all_links = link.interface_set.all()
            ip_size = len(all_links)
            # up ip_size with one if self is not in all_links
            if not [ x for x in all_links if x.id == self.instance.pk ]:
                ip_size += 1 
            return calc_subnet(ip_size)

    def clean_ssid(self):
        self.type = self.cleaned_data.get('type')
        edit = bool(self.instance.pk)
        node = self.cleaned_data.get('node')
        iface = self.cleaned_data.get('iface')
        accesspoint = self.cleaned_data.get('accesspoint')
        direction = self.cleaned_data.get('direction')
        link = self.cleaned_data.get('link')
        orig_ssid = self.cleaned_data.get('ssid')
        
        if not link_is_wireless(self):
            return None

        # iface is saved for the first time
        if not edit and not link: 
            return new_ssid(node.network, node, iface, accesspoint, direction)

        # iface is in managed mode
        if not edit and link:
            return link.ssid
        if self.instance.pk != link.id:
            return link.ssid

        # iface has become master or
        # iface changed having accesspoint
        if self.instance.link_id != link.id or \
                self.instance.accesspoint != accesspoint:
            return new_ssid(node.network, node, iface, accesspoint, direction)
        
        # else, don't change ssid and return original
        return orig_ssid

    def clean_mode(self):
        edit = bool(self.instance.pk)
        link = self.cleaned_data.get('link')
        
        if not edit and not link:
            return MASTER

        if not edit and link:
            return MANAGED

        if self.instance.pk == link.id:
            return MASTER
        
        return MANAGED

class DhcpStaticForm(forms.ModelForm):
    class Meta:
        model = DhcpStatic

    def clean_hostname(self):
        hostname = self.cleaned_data.get('hostname')
        re_hostname = re.compile('[0-9a-zA-Z-]+')
        if not hostname == re_hostname.match(hostname).group():
            raise forms.ValidationError(
                    'Not a valid hostname. Use letters, digits and -.')
        return hostname

    def clean_macaddress(self):
        macaddress = self.cleaned_data.get('macaddress')
        re_mac = re.compile('([0-9a-fA-F]{2}[:]){5}[0-9a-fA-F]{2}') 
        if not re_mac.match(macaddress):
            raise forms.ValidationError( "Please enter a valid macaddress in"
                    " the form of : XX:XX:XX:XX:XX:XX")
        return macaddress

    def clean_accesspoint(self):
        accesspoint = self.cleaned_data.get('accesspoint')
        if not accesspoint.accesspoint:
            raise forms.ValidationError("Please choose an accesspoint that is" 
                    " enabled as an accesspoint")
        return accesspoint

    def clean_ipaddress(self):
        # XXX: don't forget to update this address when changing ips.
        accesspoint = self.cleaned_data.get('accesspoint')
        ip = accesspoint.ip
        netmask = accesspoint.netmask
        ip_size = int(pow(2,32-netmask))
        networkaddrL = network(ip, netmask)
        # 1, -1 compensate for network and broadcast addr. 
        iplist = [show_addr(networkaddrL + i) for i in xrange(1, ip_size-1)]
        for i in accesspoint.interface_set.all():
            iplist.remove(i.ip)
        [iplist.remove(i.ipaddress) for i in 
                accesspoint.dhcpstatic_set.all() if not 
                i.pk == self.instance.pk]
        iplist.reverse()
        return iplist.pop()
