from django import forms
from exodus.models import Location, Node, Interface
from exodus.wllogic import free_master_ip, link_has_compat_type, \
        link_is_wireless, new_ssid
from exodus.wllogic import MASTER, MANAGED, calc_subnet
from exodus.settings import AP_NETMASK
from exodus.wlipcalc import IPCalc

class LocationForm(forms.ModelForm):
    class Meta:
        model = Location

class NodeForm(forms.ModelForm):
    class Meta:
       model = Node

    def clean_masterip(self):
        new_network = self.cleaned_data.get('network')
        # self.instnace.pk is only available when Node has been save before.
        if self.instance.pk is None:
            masterip = free_master_ip(new_network)
        else:
            # check if network has changed
            old_network = Node.objects.get(pk=self.instance.pk).network
            if old_network == self.cleaned_data.get('network'):
                masterip = self.cleaned_data.get('masterip')
            else:
                #XXX: Need to update ipconfigurations for depending interfaces
                #XXX: Need to set netmask for interfaces
                masterip = free_master_ip(new_network) 
        return masterip        

class InterfaceForm(forms.ModelForm):
    class Meta:
        model = Interface

    def clean_polar(self):
        polar = self.cleaned_data.get('polar')
        self.type = self.cleaned_data.get('type')
        
        if polar and not link_is_wireless(self):
            raise forms.ValidationError( 
                    "An ethernet interface can't have a polarization.")
        else:
            return polar

    def clean_antenna(self):
        self.type = self.cleaned_data.get('type')
        antenna = self.cleaned_data.get('antenna')
        if antenna and not link_is_wireless(self):
            raise forms.ValidationError(
                    "An ethernet interface can't have an antenna.")
        return antenna

    def clean_link(self):
        link = self.cleaned_data.get('link')
        # the type of the interface, eth, 11a, 11b, 11g.
        type = self.cleaned_data.get('type')

        # self.instance raises DoesNotExist, 
        # but self.instance.[pk, ssid, polar, etc] doesn't
        edit = bool(self.instance.pk)

        if link:
            # if link is to self we don't need to check anything else
            if self.instance.pk == link.pk:
                return link

            # check if the two links are compatible
            elif not link_has_compat_type(link.type, type):
                raise forms.ValidationError( 
                    'Link type %s is not compatible with %s' 
                    %(link.type, type))

            # Link can't be to same node, unless link is to self, 
            # which we checked above.
            elif self.instance.node_id == link.node_id:
                raise forms.ValidationError(
                    "A link can't be to another interface on the same node")

            elif self.cleaned_data.get('accesspoint') and \
                    self.instance.pk != link.pk:
                raise forms.ValidationError( "A link can't be made to another interface when this interface has an accesspoint.")

            # if link is referenced to itself, link is master and linkable
            elif link.link.pk == link.pk:
                return link

            # if link is ethernet, don't worry about master and managed.
            # but we do want every link to link to one 'master' in the
            # subnet.
            # We assume that the link of a link is always the master.
            # if link is to self, link.link_id == link.id
            elif not link_is_wireless(link):
                if link.id == link.link_id:
                    return link
                else:
                    return link.link

            # if this elif is True, it means that 'link' is in managed 
            # mode but the link of link can change modes (master/managed).
            # And check if that master has only 2, including himself, links
            # and is not an accesspoint himself.
            # We don't update the netmasks because 2 links should
            # already have a /30, and the number of links doesn't change
            # here.
            elif len(link.link.interface_set.all()) == 2 and not \
                    link.link.accesspoint:

                # Define the new master and the new slave
                new_master = link
                new_slave = new_master.link
                # set both links to the new master
                new_master.link = new_master
                new_slave.link = new_master
                #update mode(manage/master) when an interface is wireless
                if not link_is_wireless(new_master):
                    new_master.type = MASTER
                    new_slave.type = MANAGED 
                # update ssids
                ssid = new_ssid(new_master)
                new_master.ssid = ssid
                new_slave.ssid = ssid

                # calc the new ipaddresses
                ip = IPCalc(new_master.node, 30)  
                new_master.ip, new_slave.ip = ip.ips
                new_master.netmask = new_slave.netmask = 30
                    
                # Save 
                new_master.save()
                new_slave.save()
                return link

            elif len(link.link.interface_set.all()) > 2 or \
                    link.link.accesspoint:
                raise forms.ValidationError(
                    "The other interface is in managed mode, and can't be changed to master.") 
            # We shouldn't come here, because all link possibilities are 
            # accounted for.
            else:
                raise forms.ValidationError(
                    "This error should never come up, please report this bug.")
        # If this is an edit and link is not defined, we let link reference 
        # to itself.
        elif edit:
            link = Interface.objects.get(pk=self.instance.pk)
            return link
        # This is a new interface, can't be linked to itself because it hasn't 
        # been saved.
        else:
            return link

    def clean_ip(self):
        """Cleans the ip.

        We calculate the ipaddresses of this link and the ones linked to it.
        We also define the netmask in this routine.
        """
        node = self.cleaned_data.get('node')
        ip = self.cleaned_data.get('ip')
        link = self.cleaned_data.get('link')   
        ap = self.cleaned_data.get('accesspoint')
        edit = bool(self.instance.pk)

        # if we have an error, we don't calculate new ip addresses.
        if self._errors:
            raise forms.ValidationError(
                "Please correct the errors in the form.")

        if not link:
            if not ap:
                self.data['netmask'] = 32
                try:
                    new_ip = IPCalc(node, 32)
                except Exception:
                    raise forms.ValidationError(
                        "Not enough free ips for this node.")
                return new_ip.ips[0]

            if ap:
                self.data['netmask'] = AP_NETMASK       
                try:
                    new_ip = IPCalc(node, AP_NETMASK)
                except Exception:
                    raise forms.ValidationError(
                        "Not enough free ips for this node.")
                return new_ip.ips[0]

        else:
            pk = self.instance.pk
            # link is managed
            if link.id != pk:
                all_links = link.interface_set.all()
                ip_size = len(all_links)
                # up ip_size with one if self is not in all_links
                if not [ i for i in all_links if i.id == pk ]:
                    ip_size += 1 
                if link.accesspoint:
                    netmask = AP_NETMASK
                else:
                    netmask = calc_subnet(ip_size)

                try:
                    new_ip = IPCalc(link.node, netmask, link.id)
                except Exception:
                    raise forms.ValidationError(
                        "Not enough free ips for this node.")

                new_ip.ips.reverse()
                link.ip = new_ip.ips.pop()
                link.netmask = netmask
                link.save()
                ip = new_ip.ips.pop()
                self.data['netmask'] = netmask
                # filter out own and master interface
                for i in (i for i in all_links if not i.id in (pk, link.id)):
                    i.ip = new_ip.ips.pop()
                    i.netmask = netmask
                    i.save()
                return ip

            # link is master 
            if link.id == self.instance.pk:
                all_links = link.interface_set.all()
                if ap:
                    netmask = AP_NETMASK
                else:
                    ip_size = len(all_links)
                    # up ip_size with one if self is not in all_links
                    if not [ x for x in all_links if x.id == pk ]:
                        ip_size += 1 
                    netmask = calc_subnet(ip_size)

                try:
                    new_ip = IPCalc(link.node, netmask, link.id)
                except Exception:
                    raise forms.ValidationError(
                        "Not enough free ips for this node.")
                new_ip.ips.reverse()
                ip = new_ip.ips.pop()
                self.data['netmask'] = netmask
                for i in (i for i in all_links if not i.id == pk):
                    i.ip  = new_ip.ips.pop()
                    i.netmask = netmask
                    i.save() 
                return ip

    #XXX: change SSID when iface.name changes    
