from django import forms
from exodus.models import Node, Interface
from exodus.wllogic import free_master_ip, link_has_compat_type, \
        link_is_wireless, new_ssid
from exodus.wllogic import MASTER, MANAGED

class NodeForm(forms.ModelForm):
    class Meta:
       model = Node

    def clean_masterip(self):
        new_network = self.cleaned_data.get('network')
        # self.instnace.pk is only available when Node has been save before.
        if self.instance.pk is None:
            masterip = free_master_ip(new_network)
        else:
            # check if network has changed
            old_network = Node.objects.get(pk=self.instance.pk).network
            if old_network == self.cleaned_data.get('network'):
                masterip = self.cleaned_data.get('masterip')
            else:
                #XXX: Need to update ipconfigurations for depending interfaces
                #XXX: Need to set netmask for interfaces
                masterip = free_master_ip(new_network) 
        return masterip        

class InterfaceForm(forms.ModelForm):
    class Meta:
        model = Interface

    def clean_link(self):
        link = self.cleaned_data.get('link')
        # the type of the interface, eth, 11a, 11b, 11g.
        type = self.cleaned_data.get('type')

        # self.instance raises DoesNotExist, but self.instance.[pk, ssid, polar, etc] doesn't
        # Primary Keys's always start at 1.
        if self.instance.pk:
            edit = True 
        else:
            edit = False
            
        if link:
            # if link is to self we don't need to check anything else
            if self.instance.pk == link.pk:
                return link
            # Link can't be to same node, unless link is to self, 
            # which we checked above.
            elif self.instance.node_id == link.node_id:
                raise forms.ValidationError(
                    "A link can't be to another interface on the same node")

            elif self.cleaned_data.get('accesspoint') and \
                    self.instance.pk != link.pk:
                raise forms.ValidationError( "A link can't be made to another interface when this interface has an accesspoint.")

            # check if the two links are compatible
            elif not link_has_compat_type(link.type, type):
                raise forms.ValidationError( 
                    'Link type %s is not compatible with %s' 
                    %(link.type, type))

            # if link is referenced to itself, link is master and linkable
            elif link.link.pk == link.pk:
                return link

            # if link is ethernet, don't worry about master and managed.
            elif not link_is_wireless(link):
                return link

            # if this elif is True, it means that 'link' is in managed 
            # mode but the link of link can change modes (master/managed).
            # And check if that master has only 2, including himself, links
            # and is not an accesspoint himself.
            # We don't update the netmasks because 2 links should
            # already have a /30, and the number of links doesn't change
            # here.
            elif len(link.link.interface_set.all()) == 2 and not \
                    link.link.accesspoint:

                # Define the new master and the new slave
                new_master = link
                new_slave = new_master.link
                # set both links to the new master
                new_master.link = new_master
                new_slave.link = new_master
                #update mode(manage/master) when an interface is wireless
                if not link_is_wireless(new_master):
                    new_master.type = MASTER
                    new_slave.type = MANAGED 
                # update ssids
                ssid = new_ssid(new_master)
                new_master.ssid = ssid
                new_slave.ssid = ssid
                #XXX: do ip address switch stuff
                # 
                # Save 
                new_master.save()
                new_slave.save()
                return link

            elif len(link.link.interface_set.all()) > 2 or \
                    link.link.accesspoint:
                raise forms.ValidationError(
                    "The other interface is in managed mode, and can't be changed to master.") 
            # We shouldn't come here, because all link possibilities are 
            # accounted for.
            else:
                raise forms.ValidationError(
                    "This error should never come up, please report this bug.")
        # If this is an edit and link is not defined, we let link reference 
        # to itself.
        elif edit:
            link = Interface.objects.get(pk=self.instance.pk)
            return link
        # This is a new interface, can't be linked to itself because it hasn't 
        # been saved.
        else:
            return link

    #XXX: change SSID when iface.name changes    
