import unittest

class AddTest(unittest.TestCase):
    def setUp(self):
        #Set up the client
        from django.test.client import Client
        self.client = Client()
        
        #Set up a inmemory database
        from exodus import settings
        self.olddbname = settings.DATABASE_NAME
        from django.db import connection
        self.dbname = connection.creation.create_test_db(0)
        
        from exodus.models import Location

        self.location = Location(description = 'Hooigracht', \
            longitude = '45.675433', latitude = '56.543332')
        self.location.save()

    def tearDown(self):
        from django.db import connection
        connection.creation.destroy_test_db(self.olddbname, 0)


class wllogic(unittest.TestCase):
    def setUp(self):
        #Set up the client
        from django.test.client import Client
        self.client = Client()
        
        #Set up the database
        from django.db import connection
        from exodus import settings
        self.olddbname = settings.DATABASE_NAME
        self.dbname = connection.creation.create_test_db(0)

    def tearDown(self):
        from django.db import connection
        connection.creation.destroy_test_db(self.olddbname, 0)

    def test_new_ssid(self):
        class nic(object):
            iface = 'ath0'     
            class node(object):
                name = 'CeTIM'
                class network(object):
                    name = 'wleiden.net'
        from exodus.wllogic import new_ssid
        ssid = new_ssid(nic())        
        
        self.failUnlessEqual(ssid, 'ath0.CeTIM.wleiden.net')

    def test_parse_show_addr(self):
        ip = '172.16.2.0'
        from exodus.wllogic import parse_addr
        parsed = parse_addr(ip)
        self.failUnlessEqual(str(parsed), '2886730240')

        from exodus.wllogic import show_addr
        o_ip = show_addr(parsed)
        self.failUnlessEqual(o_ip, ip)

        # when using an address larger then 256, move 1 address up.
        ip = '172.16.2.256'
        o_ip = show_addr(parse_addr(ip))
        self.failUnlessEqual(o_ip, '172.16.3.0')

    def test_netmask2subnet(self):
        from wllogic import netmask2subnet

        self.failUnlessRaises(ValueError, netmask2subnet, 33)
        self.failUnlessRaises(ValueError, netmask2subnet, -1)

        valid_subnet = netmask2subnet(30)
        self.failUnlessEqual(str(valid_subnet), '17179869180')
    
    def test_get_subnet(self):
        from wllogic import get_subnet
        self.failUnlessRaises(ValueError, get_subnet, -1)
        self.failUnlessRaises(ValueError, get_subnet, 33)
        
        valid_netmask = get_subnet(30)
        self.failUnless(valid_netmask, '255.255.255.252')
    
    def test_network(self):
        from wllogic import network
        from wllogic import show_addr
        network_addr = show_addr(network('172.18.5.10', 24))
        self.failUnlessEqual(network_addr, '172.18.5.0')

    def test_broadcast(self):
        from wllogic import broadcast
        from wllogic import show_addr
        
        broadcast_addr = show_addr(broadcast('172.16.5.232', 24))
        self.failUnlessEqual(broadcast_addr, '172.16.5.255')

        self.failUnlessRaises(ValueError, broadcast, '172.16.5.1', -1)
        self.failUnlessRaises(ValueError, broadcast, '172.16.5.1', 33)
    
    def test_get_network(self):
        from wllogic import get_network
        network_addr = get_network('172.17.2.2', 30)
        self. failUnlessEqual(network_addr, '172.17.2.0')

        self.failUnlessRaises(ValueError, get_network, '172.17.2.0', -1)
        self.failUnlessRaises(ValueError, get_network, '172.17.2.0', 33)

    def test_get_broadcast(self):
        from wllogic import get_broadcast
        broadcast_addr = get_broadcast('172.17.2.2', 24)
        self.failUnlessEqual(broadcast_addr, '172.17.2.255')
        
        self.failUnlessRaises(ValueError, get_broadcast, '172.17.2.0', -1)
        self.failUnlessRaises(ValueError, get_broadcast, '172.17.2.0', 33)

    def test_free_master_ip(self):
        from wllogic import free_master_ip
        from exodus.models import Network, Node
        nodes = Node.objects.all()
        ip_list = []
        for i in nodes:
            ip_list.append(i.masterip)
        network = Network.objects.get(pk=1)
        ip = free_master_ip(network) 
        self.failIf(ip in ip_list)
        self.failUnlessEqual(ip, '172.16.2.1')

    def test_calc_subnet(self):
        from wllogic import calc_subnet
        self.failUnlessEqual(calc_subnet(1), 32)
        self.failUnlessRaises(ValueError, calc_subnet, 0)
        self.failUnlessRaises(ValueError, calc_subnet, 255)

        l = []
        for i in range(1,255):
            l.append(calc_subnet(i))
        k = []
        for i in range(24,33):
            k.append(l.count(i))
        n = [128, 64, 32, 16, 8, 4, 1, 0, 1]
        self.failUnlessEqual(k,n)

    def test_CHOICES(self):
        from wllogic import MASTER, MANAGED
        self.failUnlessEqual(MASTER, 'ms')
        self.failUnlessEqual(MANAGED, 'mn')
        
class Link(unittest.TestCase):
    def setUp(self):
        class link(object):
            def __init__(self, type, node):
                self.type = type
                self.node = node

        self.link00 = link('eth', 1)
        self.link01 = link('eth', 2)
        self.link10 = link('11a', 1)
        self.link11 = link('11a', 2)
        self.link20 = link('11b', 1)
        self.link21 = link('11b', 2)
        self.link30 = link('11g', 1)		
        self.link31 = link('11g', 2)

    def test_link_has_compat_type(self):
        from wllogic import link_has_compat_type
       
        # test link to self
        self.failUnless(link_has_compat_type(self.link00.type, 
				self.link00.type))
        # test eth
        self.failUnless(link_has_compat_type(self.link00.type, 
				self.link01.type))
        # test 11a
        self.failUnless(link_has_compat_type(self.link10.type,
				 self.link11.type))
        # test 11b
        self.failUnless(link_has_compat_type(self.link20.type, 
				self.link21.type))
        # test 11g
        self.failUnless(link_has_compat_type(self.link30.type, 
				self.link31.type))
        # test 11b vs 11g
        self.failUnless(link_has_compat_type(self.link20.type, 
				self.link30.type))
        self.failUnless(link_has_compat_type(self.link30.type, 
				self.link20.type))

        # test fail eth vs 11a
        self.failIf(link_has_compat_type(self.link00, self.link10))
        # test fail eth vs 11b
        self.failIf(link_has_compat_type(self.link00, self.link20))
        # test fail eth vs 11g
        self.failIf(link_has_compat_type(self.link00, self.link30))
        # test fail 11a vs 11b
        self.failIf(link_has_compat_type(self.link10, self.link20))
        # test fail 11a vs 11g
        self.failIf(link_has_compat_type(self.link10, self.link30))
    
    def test_link_not_same_node(self):
        from wllogic import link_is_not_to_itself
        self.failUnless(link_is_not_to_itself(self.link00, self.link01))
        self.failIf(link_is_not_to_itself(self.link20, self.link30))
    
    def test_link_is_wireless(self):
        from wllogic import link_is_wireless
        self.failIf(link_is_wireless(self.link00))
        self.failUnless(link_is_wireless(self.link10))
        self.failUnless(link_is_wireless(self.link20))
        self.failUnless(link_is_wireless(self.link30))
		
def suite():
    s = unittest.TestSuite()
    s.addTest(unittest.makeSuite(AddTest))
    s.addTest(unittest.makeSuite(wllogic))
    s.addTest(unittest.makeSuite(Link))

    return s
