# (c) Roland van Laar 2006

from django.db import models
from django.http import HttpResponse, HttpResponseRedirect
from django.template import Context, loader
from django.shortcuts import render_to_response
from django.core.urlresolvers import reverse
from django.forms.formsets import formset_factory
from django import forms
from socket import gethostname 

from exodus.models import *
from exodus.wllogic import freeMasterIP, newSSIDName, addInterlinkIP, freeInterlinkIP, freePublicAPIP



def node(request, node):
	object = Node.objects.get(name=node)
	return render_to_response('node-detail.html', {'node': object})


class addLinkForm(forms.Form):
	class Meta:
		model = Node
		exclude = ( 'shortdesc', 'desc' )

def addLink(request):
	if request.POST:
		form = addLinkForm(request.POST)
		if form.is_valid():
			instance = form.save(commit=False)
			# Get this code working and clean it up afterwards
			# check if we have two different ifaces from two different nodes
			# Should add a error message if they are from the same node
			compatibleInterfaces = False
			differentNodes = True
			errorMessage = ""
			
			# check if they have the same mode
			if instance.iface1.node.id == instance.iface2.node.id:
				differentNodes = False
			
			# Interface check
			# XXX: Make sure interfaces type is defined as beeing a certain type instead of hardcoding
			if instance.iface1.type == instance.iface2.type:
				compatibleInterfaces = True
			if str(instance.iface1.type) == '11b' and str(instance.iface2.type) == '11g':
				compatibleInterfaces = True
			if str(instance.iface1.type) == '11g' and str(instance.iface2.type) == '11b':
				compatibleInterfaces = True
			
			if compatibleInterfaces and differentNodes:
				instance.save()
				return HttpResponseRedirect(reverse('nodelist'))
			else:
				form = addLinkForm(instance = instance)
				if compatibleInterfaces == False:
					errorMessage = "Please use compatible interfaces"
				elif differentNodes == False:
					errorMessage = "No interlink possible between interfaces on the same node"
				
				return render_to_response('addinterlink.html',{'form': form, 'error_message' : errorMessage })
			#XXX :if wifi: check master and managed states for existings NICs/Interlinks
			#XXX :check if omni, than Nic must be in master mode
			#Generate SSID
			# add descriptions
			#Generate ips and netmask
			#Save form
	else:
		#XXX: Autofetch new IP numbers
		newForm = Link()
		form = addLinkForm(instance = newForm)
	return render_to_response('addinterlink.html',{'form': form})


class GenericHandler():
	"""Conventions used: type has to to the same name as the dependency object, delet template is named delete<type>.html"""
	def __init__(self, request, mode):
		# Strip 'Handler' of name
		type = self.__class__.__name__[:-7:]
		formClass = eval(type + 'Form')

		self.object = eval('self.' + type[0].lower() + type[1::])
		self.title = self.object._meta.verbose_name
		self.request = request
		
		if request.POST.has_key('cancel'):
			self.response = HttpResponseRedirect(reverse('node-detail', args=[self.node.name]))
		elif request.POST.has_key('proceed'):
			if mode == 'delete':
				self._deleteete()
				self.response = HttpResponseRedirect(reverse('node-detail', args=[self.node.name]))
			else:
				# First checking whether form is valid, then add/edit actions
				self.form = formClass(request.POST, instance=self.object)
				if self.form.is_valid():
					# Set response on forehand, to allow override in procedure
					self.response = HttpResponseRedirect(reverse('node-detail', args=[self.node.name]))
					if mode == 'add':
						self._add()
					elif mode == 'edit':
						self._edit()
				else:
					message = 'Form error, please edit and resubmit'
					self.response = render_to_response('genericForm.html', {'form': self.form,
						'message' : message, 'title' : self.title, 'mode' : mode,
						'type' : type, 'object': self.object })
		else:
			message = 'Please edit and submit'
			
			# Dirty? hack to allow initial form to be filled with date for GET request, no errors raised
			if request.GET and mode == 'add':
				self.form = formClass(request.GET, instance=self.object)
				self.form._errors = {}
			else:
				self.form = formClass(instance=self.object)

			self.response = render_to_response('genericForm.html', {
				'form': self.form, 'message' : message, 'title' : self.title,
				'mode' : mode, 'type' : type, 'object': self.object,
				'delInclude' : "delete" + type.capitalize() + ".html",
				'addInclude' : "add" + type.capitalize() + ".html",
				'editInclude' : "edit" + type.capitalize() + ".html" })

	def _add(self):
		self.form.save()

	def _edit(self):
		self.form.save()

	def _deleteete(self):
		self.object.delete()
	
	def render_to_response(self):
		return self.response


#
# PublicAP
class PublicAPForm(forms.ModelForm):
	class Meta:
		model = PublicAP
		exclude = ('shortdesc', 'desc', 'ip', 'dhcpstart', 'dhcpstop')

class PublicAPHandler(GenericHandler):
	def __init__(self, request, node, interface, publicAP, mode):
		self.node = Node.objects.get(name=node)
		self.interface = Interface.objects.get(node=self.node, iface=interface)
		if mode == 'add':
			self.publicAP = PublicAP(iface=self.interface)
		else:
			self.publicAP =  PublicAP.objects.get(iface=self.interface, pk=publicAP)
		GenericHandler.__init__(self, request, mode)


	def _add(self):
		_instance = self.form.save(commit=False)
		# Find IP range inside interface range with disired size/subnet
		_instance.ip = freePublicAPIP(_instance.iface, _instance.netmask)
		_instance.dhcpstart = 1
		_instance.dhcpstop = 2
		# If wireless generate ssid name
		_instance.ssid = newSSIDName(_instance.iface.node, _instance.iface, 'omni')
		_instance.save()

def genericPublicAP(request, node, interface, publicAP, mode):
	handler = PublicAPHandler(request, node, interface, publicAP, mode)
	return handler.render_to_response()



#
# Interface 
class InterfaceForm(forms.ModelForm):
	class Meta:
		model = Interface
		exclude = ( 'ip', 'ssid', 'mode', 'channel', 'shortdesc' )

class InterfaceHandler(GenericHandler):
	def __init__(self, request, node, interface, mode):
		self.node = Node.objects.get(name=node)
		if mode == 'add':
			self.interface = Interface(node=self.node)
		else:
			self.interface = Interface.objects.get(node=self.node, iface=interface)
		GenericHandler.__init__(self, request, mode)

	def _add(self):
		self._saveInterface()

	def _edit(self):
		self._saveInterface()

	def _saveInterface(self):
		#XXX: Should have checking wether form has actually been changed to avoid creation of IPs every time
		_instance  = self.form.save(commit=False)
		if str(_instance.type) != "eth":
			_instance.ssid = newSSIDName(_instance.node, _instance.iface, 'unused')
			_instance.channel = '1'
			_instance.mode = 1 # set to master
		if not _instance.link:
			_instance.ip = freeInterlinkIP(_instance)
		else:
			_instance.ip = addInterlinkIP(_instance.link)
		_instance.save()
		#Dirty to hack to get reference to self working
		if not _instance.link:
			_instance.link = _instance
			_instance.save()


def genericInterface(request, node, interface, mode):
	handler = InterfaceHandler(request, node, interface, mode)
	return handler.render_to_response()

#
# Node
class NodeForm(forms.ModelForm):
	class Meta:
		model = Node
		exclude = ( 'masterip' )

class NodeHandler(GenericHandler):
	def __init__(self, request, node, mode):
		if mode == 'add':
			self.node = Node()
		else:
			self.node = Node.objects.get(name=node)
		GenericHandler.__init__(self, request, mode)

	def _add(self):
		# input a valid master ip into new_data
		_instance  = self.form.save(commit=False)
		_instance.masterip = freeMasterIP(_instance)
		_instance.save()

	def _delete(self):
		for _master in Interface.objects.filter(node=_instance):
			if _master.link == _master:
				for _makeMaster in Interface.objects.filter(link=_master):
					_makeMaster.link = _makeMaster
					_makeMaster.save()
		_instance.delete()
		# As node is deleted, goto overview page
		self.response = HttpResponseRedirect(reverse('node-overview'))

def genericNode(request, node, mode):
	handler = NodeHandler(request, node, mode)
	return handler.render_to_response()

#
# Location
class LocationForm(forms.ModelForm):
	class Meta:
		model = Location

class LocationHandler(GenericHandler):
	def __init__(self, request, location, mode):
		if mode == 'add':
			self.location = Location()
		else:
			self.location = Location.objects.get(description=location)
		GenericHandler.__init__(self, request, mode)

	def _add(self):
		self._saveInterface()
		# After adding a location, allow adding a Node with this location
		self.response = HttpResponseRedirect(reverse('exodus.views.genericNode', args=["add", "new"]) + "?location=%i" % self._instance.pk)

	def _delete(self):
		self.object.delete()
		self.response = HttpResponseRedirect(reverse('node-overview'))

def genericLocation(request, location, mode):
	handler = LocationHandler(request, location, mode)
	return handler.render_to_response()




def nodename2id(node):
	"""Convert node name to ID if found in the database"""
	return Node.objects.get(name=node).id

def conf_generator(node, template):
	object_id = nodename2id(node)
	object = Node.objects.get(pk=object_id)

	#Some extra values for statictics info
	object.name = Node.objects.get(pk=object_id).name
	object.host = gethostname()
#	object.nic = object.nic_set.all()
	
	#Used for named.conf
	object.domains = Network.objects.all()

	t = loader.get_template(template)
	c = Context({'object': object})
	return HttpResponse(t.render(c), mimetype='text/plain')

def conf_dhcpd_conf(request, object_id):
	"""isc-dhcpd-3-server conf generator"""
	return conf_generator(object_id, 'dhcpd.conf')

def conf_named_conf(request, object_id):
	"""bind conf generator"""
	return conf_generator(object_id, 'named.conf')

def conf_rc_local(request, object_id):
    """5.X legacy rc.local conf generator"""

    object_id = nodename2id(object_id)
    object = Node.objects.get(pk=object_id)
    object.name = Node.objects.get(pk=object_id).name
    object.host = gethostname()

    # check if it is a wireless nic, and add wifimode for wicontrol  
    for nic in object:
        if nic.ssid:
            if nic.mode.mode == 'master':
                nic.wifimode = 6
                nic.wifidesc = 'master mode'
            else:
                nic.wifimode = 1
                nic.wifidesc = 'managed mode'
    t = loader.get_template('rc-local.txt')
    c = Context({'object': object, 
        })
    return HttpResponse(t.render(c), mimetype='text/plain')

def conf_rc_node_local5X(request, object_id):
	"""5.X Legacy rc.node.local conf generator"""

	object_id = nodename2id(object_id)

	object.name = Node.objects.get(pk=object_id).name
	object.host = gethostname()

	t = loader.get_template('rc-node-local.txt')
	c = Context({'object': object,
	    })
	return HttpResponse(t.render(c), mimetype='text/plain')

def conf_rc_node_local(request, object_id):
	"""rc.node.local conf generator"""
	
	object_id = nodename2id(object_id)
	
	object = Node.objects.get(pk=object_id).interface_set.all()
	object.name = Node.objects.get(pk=object_id).name
	object.host = gethostname()

	t = loader.get_template('rc-node-local.txt')
	c = Context({'object': object, })

	print c
	return HttpResponse(t.render(c), mimetype='text/plain')


def conf_resolv_conf(request, object_id):
	"""resolv.conf generator"""
	return conf_generator(object_id, 'resolv.conf')

