#!/usr/bin/env python
# Scan Wireless Leiden Network and report status of links and nodes
#
# Rick van der Zwet <info@rickvanderzwet.nl>

from pprint import pprint
from xml.dom.minidom import parse, parseString
import os.path
import re
import subprocess
import sys
import time
import yaml

SCAN_RANGE='172.16.0.0/21'


def get_attribute(node,attr):
  return node.attributes[attr].value

def attribute_from_node(parent,node,attr):
  return parent.getElementsByTagName(node)[0].attributes[attr].value

def parse_port(node):
  item = dict()
  item['protocol'] = get_attribute(node,'protocol')
  item['portid'] = get_attribute(node,'portid')
  item['state'] = attribute_from_node(node,'state','state')
  item['reason'] = attribute_from_node(node,'state','reason')
  return item

def parse_ports(node):
  item = dict()
  for port in node.getElementsByTagName('port'):
    port_item = parse_port(port)
    item[port_item['portid']] = port_item
  return item

def parse_host(node):
  # Host status
  item = dict()
  item['state'] = attribute_from_node(node,'status','state')
  item['reason'] = attribute_from_node(node,'status','reason')
  item['addr'] = attribute_from_node(node,'address','addr') 
  item['addrtype'] = attribute_from_node(node,'address','addrtype')

  # Service status
  ports = node.getElementsByTagName('ports')
  if ports:
    item['port'] = parse_ports(ports[0])
  return item

def parse_nmap(root):
  status = dict()
  for node in root.childNodes[2].getElementsByTagName('host'):
    scan = parse_host(node)
    if not status.has_key(scan['addr']):
      status[scan['addr']] = scan
  return status


def _run_nmap(command, result_file="-"):
  """ Run/Read nmap XML with various choices"""

  command = "nmap -n -oX - %s" %(command)
  dom = None
  if result_file == "-":
    print "# New run '%s' results not saved" % (command)
    dom = parseString(subprocess.Popen(command.split(),
            stdout=subprocess.PIPE).communicate()[0])
  elif os.path.exists(result_file) and os.path.getsize(result_file) > 0:
    print "# Reading stored NMAP results from '%s'" % (result_file)
    dom = parse(result_file)
  else:
    print "# New run '%s' saving results in %s" % (command, result_file)
    print "# Waiting for nmap run to complete"
    f = open(result_file,'w')
    p = subprocess.Popen(command.split(),stdout=f)
    while p.poll() == None:
      print ".",
      sys.stdout.flush()
      time.sleep(1)
    print "DONE"
    if p.returncode != 0:
      print "# ERROR in nmap command"
      sys.exit(1) 
    f.close()
    dom = parse(result_file)
  return parse_nmap(dom)

def do_nmap(command, iphosts, result_file="-"):
  """ Wrapper around _run_nmap to get listing of all hosts"""
  # Get all hosts to be processed
  status = _run_nmap("-sL " + iphosts)
  status.update(_run_nmap("%s %s" % (command, iphosts), result_file))
  return status

def do_snmpwalk(host, oid):
   """ Do snmpwalk, returns (p, stdout, stderr)"""
   # GLobal SNMP walk options
   snmpwalk = ('snmpwalk -r 0 -t 1 -OX -c public -v 2c %s' % host).split()
   p = subprocess.Popen(snmpwalk + [oid],
       stdout=subprocess.PIPE, stderr=subprocess.PIPE)
   (stdoutdata, stderrdata) = p.communicate()
   stdoutdata = stdoutdata.split('\n')[:-1]
   stderrdata = stderrdata.split('\n')[:-1]
   return (p, stdoutdata, stderrdata)


def run_scan(host_filter=None):
  """ Big working processing all scanning work"""
  # Do a NMAP discovery
  if host_filter:
    iphosts = " ".join(host_filter)
    result_file = "-"
  else:
    iphosts = SCAN_RANGE
    result_file = '/tmp/test.xml'

  status = do_nmap("-p T:ssh,U:domain,T:80,T:ntp,U:snmp,T:8080 -sU -sT ",iphosts, result_file)
  mac_to_host = dict()
  host_processed = dict()
  stored_status = dict()

  #
  # Gather SNMP data from hosts
  for host, scan in status.iteritems():
    if scan['state'] != "up":
      continue
    
    # Filter set? use it
    if host_filter and not host in host_filter:
      continue
  
    print '# Processing host %s' % host
    # IP -> Mac addresses found in host ARP table, with key IP
    status[host]['arpmac'] = dict()
    # MAC -> iface addresses, with key MAC
    status[host]['mac'] = dict()
    # Mirrored: iface -> MAC addresses, with key interface name
    status[host]['iface'] = dict()
    try:
      if stored_status[host]['snmp_retval'] != 0:
        print "## SNMP Connect failed last time, ignoring"
        continue
    except:
      pass
  
    stored_status[host] = dict()
    if not "open" in scan['port']['161']['state']:
      print "## [ERROR] SNMP port not opened"
      continue
  
    (p, output, stderrdata) = do_snmpwalk(host, 'SNMPv2-MIB::sysDescr')
    stored_status[host]['snmp_retval'] = p.returncode
    # Assume host remain reachable troughout all the SNMP queries
    if p.returncode != 0:
       print "## [ERROR] SNMP failed '%s'" % ",".join(stderrdata)
       continue
  
    # Get some host details
    # SNMPv2-MIB::sysDescr.0 = STRING: FreeBSD CNodeSOM1.wLeiden.NET
    # 8.0-RELEASE-p2 FreeBSD 8.0-RELEASE-p2 #2: Fri Feb 19 18:24:23 CET 2010
    # root@80fab2:/usr/obj/nanobsd.wleiden/usr/src/sys/kernel.wleiden i386
    status[host]['sys_desc'] = output[0]
    hostname = output[0].split(' ')[4]
    release = output[0].split(' ')[5]
    stored_status[host]['hostname'] = status[host]['hostname'] = hostname
    stored_status[host]['release'] = status[host]['release'] = release
    print "## %(hostname)s - %(release)s" % stored_status[host]
  
    # Check if the host is already done processing
    # Note: the host is marked done processing at the end
    if host_processed.has_key(hostname):
      print "## Host already processed this run"
      continue  
    
    # Interface list with key the index number
    iface_descr = dict()
    # IF-MIB::ifDescr.1 = STRING: ath0
    r = re.compile('^IF-MIB::ifDescr\[([0-9]+)\] = STRING: ([a-z0-9]+)$')
    (p, output, stderrdata) = do_snmpwalk(host, 'IF-MIB::ifDescr')
    for line in output:
      m = r.match(line)
      iface_descr[m.group(1)] = m.group(2)
  
    # IF-MIB::ifPhysAddress[1] = STRING: 0:80:48:54:bb:52
    r = re.compile('^IF-MIB::ifPhysAddress\[([0-9]+)\] = STRING: ([0-9a-f:]*)$')
    (p, output, stderrdata) = do_snmpwalk(host, 'IF-MIB::ifPhysAddress')
    for line in output:
      m = r.match(line)
      # Ignore lines which has no MAC address
      if not m.group(2): continue
      index = m.group(1)
      # Convert to proper MAC
      mac = ":".join(["%02X" % int(x,16) for x in m.group(2).split(':')])
      print "## Local MAC %s [index:%s] -> %s" % (iface_descr[index], index, mac)
      status[host]['mac'][mac] = iface_descr[index]
      status[host]['iface'][iface_descr[index]] = mac
      mac_to_host[mac] = hostname
  
    # Process host SNMP status
    (p, output, stderrdata) = do_snmpwalk(host, 'RFC1213-MIB::atPhysAddress')
    # RFC1213-MIB::atPhysAddress[8][1.172.21.160.34] = Hex-STRING: 00 17 C4 CC 5B F2 
    r = re.compile('^RFC1213-MIB::atPhysAddress\[[0-9]+\]\[1\.([0-9\.]+)\] = Hex-STRING: ([0-9A-F\ ]+)$')
    for line in output:
      m = r.match(line)
      ip = m.group(1)
      # Replace spaces in MAC with :
      mac = ":".join(m.group(2).split(' ')[:-1])
      status[host]['arpmac'][ip] = mac
  
      local = '[remote]'
      if mac in status[host]['mac'].keys():
        local = '[local]'
      print "## Arp table MAC %s -> %s %s" % (ip, mac, local)
  
    # Make sure we keep a record of the processed host
    host_processed[hostname] = status[host]

  stored_status['host_processed'] = host_processed
  stored_status['mac_to_host'] = mac_to_host
  return stored_status
  

def usage():
    print "Usage: %s <all|force|stored|host HOST1 [HOST2 ...]>" % sys.argv[0]
    sys.exit(0)

stored_status = dict()
stored_status_file = '/tmp/stored_status.yaml'
if len(sys.argv) > 1:
  if sys.argv[1] == "all":
    stored_status = run_scan()
    stream = file(stored_status_file,'w')
    print "##Stored data hints to '%' " % stored_status
    stored_status = yaml.dump(stored_status, stream, default_flow_style=False)
  elif sys.argv[1] == "force":
    stored_status = run_scan()
  elif sys.argv[1] == "host":
    stored_status = run_scan(sys.argv[2:])
  elif sys.argv[1] == "stored":
    # Load data hints from previous run if exists
    if os.path.exists(stored_status_file) and os.path.getsize(stored_status_file) > 0:
      stream = file(stored_status_file,'r')
      stored_status = yaml.load(stream)
    else:
      print "[ERROR] '%s' does not exists" % stored_status_file
  else:
    usage()
else:
   usage()


host_processed = stored_status['host_processed']
mac_to_host = stored_status['mac_to_host']
# Correlation mapping
for host, details in host_processed.iteritems():
  print "# Working on %s" % host
  for ip, arpmac in details['arpmac'].iteritems():
    if arpmac in details['mac'].keys():
      # Local MAC address
      continue
    if not mac_to_host.has_key(arpmac):
      print "## [WARN] No parent host for MAC %s (%s) found" % (arpmac, ip)
    else:
      print "## Interlink %s - %s"  % (host, mac_to_host[arpmac])
        
    
