#!/bin/sh -
# Based on the idea of /etc/rc.firewall
# 
# NOTE: Consider implemention IPv6 as solution before even thinking about using
# NOTE: this (advanced) firewall rules.
#
# Firewall solution for Wireless Leiden ``iLeiden'' setup.
#
# This firewall is configured to be a 2-NAT solution, to be used in a setup
# when this box can reach the INET via the HOSTERNET network, connected on
# ``externalif''. And the WLNET network is connected on ``internalif''.
#
# a) This will provide rate-limited NAT support for the WLNET to specific ports
#    on the internet.
# b) Provide NAT support for HOSTERNET machines to access WLNET.
# c) Secures the HOSTERNET from abuse from the WLNET.
#
# Richard van Mansom <richard@vanmansom.net>
# Rick van der Zwet - <info@rickvanderzwet.nl>


allowed2internet="80,443"
maxconnections="10"

WLNET=172.16.0.0/12

# Suck in the configuration variables.
if [ -z "${source_rc_confs_defined}" ]; then
	if [ -r /etc/defaults/rc.conf ]; then
		. /etc/defaults/rc.conf
		source_rc_confs
	elif [ -r /etc/rc.conf ]; then
		. /etc/rc.conf
	fi
fi

setup_loopback () {
	############
	# Only in rare cases do you want to change these rules
	#
	${fwcmd} add 100 pass all from any to any via lo0
	${fwcmd} add 200 deny all from any to 127.0.0.0/8
	${fwcmd} add 300 deny ip from 127.0.0.0/8 to any
}

############
# Set quiet mode if requested
#
case ${firewall_quiet} in
[Yy][Ee][Ss])
	fwcmd="/sbin/ipfw -q"
	;;
*)
	fwcmd="/sbin/ipfw"
	;;
esac

logcount=500
case ${firewall_verbose} in
[Yy][Ee][Ss])
	log="log"
	;;
*)
	log=""
	;;
esac

###########
# Set Internal/External Interface
#
driver=`echo ${internalif} | sed 's/[0-9]*//g'`
seq=`echo ${internalif} | sed 's/[a-zA-Z]*//g'`

if [ ${seq} = 0 ]; then
  seq=`expr ${seq} \+ 1`
else
  seq=`expr ${seq} \- 1`
fi

externalif="$driver$seq"

# Get public ip
externalip=`ifconfig $externalif | awk '/inet/ { print $2 }'`
internalip=`ifconfig $internalif | awk '/inet/ { print $2 }'`

#XXX: Ugly hack, make me dynamic
HOSTERNET=${HOSTERNET:-$externalip/24}

echo "# [INFO] Internal (wleiden ) Interface: $internalif ($internalip) - $WLNET"
echo "# [INFO] External (internet) Interface: $externalif ($externalip) - $HOSTERNET"

############
# Flush out the list before we begin.
#
${fwcmd} -f flush

setup_loopback

############
# Block any traffic from WL to the hosters network (and maybe others)
for IP in ${firewall_block} 192.168.0.0/16 10.0.0.0/8 172.16.0.0/12
do
  ${fwcmd} add deny log logamount $logcount ip from any to ${IP} recv $internalif xmit $externalip setup
done

#############
# ICMP RULES
# Allow ICMP from and to me
${fwcmd} add allow $log icmp from any to me
${fwcmd} add allow $log icmp from me to any

# From ICMP only allow limited ICMP types to receive to the hosters network
${fwcmd} add allow $log icmp from $WLNET to $HOSTERNET icmptype 0
${fwcmd} add allow $log icmp from $HOSTERNET to $WLNET

# Block any other ICMP traffic
${fwcmd} add deny log logamount $logcount icmp from any to any



#############
# Stateful firewalling
${fwcmd} add check-state
${fwcmd} add allow tcp from any to any established

# Transparant proxy HTTP,HTTPS
${fwcmd} add allow $log tcp from $WLNET to any $allowed2internet setup limit src-addr $maxconnections

# Allow anything originating from me
${fwcmd} add allow $log tcp from me to any setup keep-state
${fwcmd} add allow $log udp from me to any keep-state
# Special Rules for HOSTERNET
${fwcmd} add allow $log tcp from $HOSTERNET to $WLNET setup keep-state
${fwcmd} add allow $log udp from $HOSTERNET to $WLNET keep-state

## INTERNAL INTERFACE
# TCP: ssh,domain,http,http-proxy,lvrouted
${fwcmd} add allow $log tcp from $WLNET to me 22,53,80,3128,12345 setup keep-state
# UDP: domain,ntp,snmp,lvrouted
${fwcmd} add allow $log udp from $WLNET to me 53,131,161,12345 keep-state

## EXTERNAL INTERFACE
# TCP: ssh,domain
${fwcmd} add allow $log tcp from any to me 22,53 setup keep-state
# UDP: domain,snmp
${fwcmd} add allow $log udp from $WLNET to me 53,161 keep-state

#############
# Outbound NAT setup
${fwcmd} add nat 100 $log all from $WLNET to any 80,443 out recv $internalif xmit $externalif
${fwcmd} add nat 100 $log all from any 80,443 to $externalip in recv $externalif
${fwcmd} nat 100 config $log if $externalif

#############
# Inbound NAT setup
${fwcmd} add nat 200 $log all from $HOSTERNET to $WLNET out recv $externalif xmit $internalif
${fwcmd} add nat 200 $log all from $WLNET to $internalip in recv $internalif
${fwcmd} nat 200 config $log if $internalif

#############
# Block anything else
${fwcmd} add 65000 deny log logamount $logcount ip from any to any
