<?php
/*
 * Project: NodeMap2.0
 * File: Network.class.php
 * Purpose: Contains a network, calculates everything like broadcast IP, netmask etc.
 */

class Network {
	public $netmaskNumber;		// Netmask number
	public $netmaskBinary;		// Binary value of netmask address
	public $netmaskDecimal;		// Decimal value of netmask address
	public $wildcardBinary;		// Binary value of wildcard address
	public $wildcardDecimal;	// Decimal value of wildcard address
	public $networkBinary;		// Binary value of network address
	public $networkDecimal;		// Decimal value of network address
	public $broadcastBinary;	// Binary value of broadcast address
	public $broadcastDecimal;	// Decimal value of broadcast address
	public $hostminBinary;		// Binary value of hostmin address
	public $hostminDecimal;		// Decimal value of hostmin address
	public $hostmaxBinary;		// Binary value of hostmax address
	public $hostmaxDecimal;		// Decimal value of hostmax address
	public $numberHosts;		// Number of possible hosts in this network

	/*
	 * Function: __construct (constructor)
	 * Description: Creating a new Network object
	 * Parameters: string $ip_address, int $ip_netmask
	 * Returns: -
	 */
	public function __construct($ip_address, $ip_netmask) {
		// Binary value of given IP address
		$binary = $this->binIP($ip_address);

		// Let's store the IP netmask number
		$this->netmaskNumber = $ip_netmask;

		// Netmask address
		$this->netmaskBinary = str_repeat('1', $ip_netmask) . str_repeat('0', 32 - $ip_netmask);
		$this->netmaskDecimal = $this->decIP($this->netmaskBinary);

		// Wildcard address
		$this->wildcardBinary = str_repeat('0', $ip_netmask) . str_repeat('1', 32 - $ip_netmask);
		$this->wildcardDecimal = $this->decIP($this->wildcardBinary);

		// Network address
		$this->networkBinary = substr($binary, 0, $ip_netmask) . str_repeat('0', 32 - $ip_netmask);
		$this->networkDecimal = $this->decIP($this->networkBinary);

		// Broadcast address
		$this->broadcastBinary = substr($binary, 0, $ip_netmask) . str_repeat('1', 32 - $ip_netmask);
		$this->broadcastDecimal = $this->decIP($this->broadcastBinary);

		// HostMin address
		$this->hostminBinary = substr($binary, 0, $ip_netmask) . str_repeat('0', 31 - $ip_netmask) . '1';
		$this->hostminDecimal = $this->decIP($this->hostminBinary);

		// HostMax address
		$this->hostmaxBinary = substr($binary, 0, $ip_netmask) . str_repeat('1', 31 - $ip_netmask) . '0';
		$this->hostmaxDecimal = $this->decIP($this->hostmaxBinary);

		// Numbers of hosts
		$this->numberHosts = bindec(substr($this->broadcastBinary, $ip_netmask)) - bindec(substr($this->networkBinary, $ip_netmask)) - 1;
	}

	/*
	 * Function: binIP
	 * Description: Converts a IP address like 192.168.0.1 to it's binary value
	 * Parameters: string $address
	 * Returns: string $address_binary
	 */
	function binIP($address) {
		$address_array = explode('.', $address);
		$address_binary = '';

		for ($i = 0; $i < 4; $i++) {
			// Create binary value
			$binary = decbin($address_array[$i]);

			// Trim value to the length
			$binary = trim($binary, 8);

			// Add zeros at the left to fit to $size
			$binary = str_repeat('0', 8 - strlen($binary)) . $binary;

			$address_binary .= $binary;
		}

		return $address_binary;
	}

	/*
	 * Function: decIP
	 * Description: Converts a binary IP address to it's decimal value like 192.168.0.1
	 * Parameters: string $binary
	 * Returns: string $address
	 */
	function decIP($binary) {
		$address_array = array();

		for ($i = 0; $i < 4; $i++) {
			$block = substr($binary, $i * 8, 8) . '<br/>';
			$address_array[] = bindec(intval($block));
		}

		$address = implode('.', $address_array); 

		return $address;
	}

	/*
	 * Function: compare
	 * Description: Checks if to networks are the same using network IP address
	 * Parameters: Network $otherNetwork
	 * Returns: true if the networks are the same, otherwise false
	 */
	function compare(Network $otherNetwork) {
		$check = false;

		// Compare the network address
		if ($this->networkDecimal == $otherNetwork->networkDecimal) {
			$check = true;
		}

		return $check;
	}

	/*
	 * Function: inNetwork
	 * Description: Checks if the given IP is in the IP range of the current network
	 * Parameters: string $ipDecimal
	 * Returns: true if the IP address is in the IP range, otherwise false
	 */
	function inNetwork($ipDecimal) {
		$check = true;

		// Check if given IP is bigger than our HostMin IP Address
		if (ip2long($ipDecimal) < ip2long($this->hostminDecimal)) {
			$check = false;
		}

		// Check if given IP is smaller than our HostMax IP Address
		if (ip2long($ipDecimal) > ip2long($this->hostmaxDecimal)) {
			$check = false;
		}
				
		return $check;
	}

	/*
	 * Function: toString
	 * Description: Converts the data of the network to a string
	 * Parameters: -
	 * Returns: string with all data
	 */
	function toString() {
		return '
			Netmask: ' . $this->netmaskDecimal . ' / ' . $this->netmaskBinary . '<br/>
			Wildcard: ' . $this->wildcardDecimal . ' / ' . $this->wildcardBinary . '<br/>
			Network: ' . $this->networkDecimal . ' / ' . $this->networkBinary . '<br/>
			Broadcast: ' . $this->broadcastDecimal . ' / ' . $this->broadcastBinary . '<br/>
			HostMin: ' . $this->hostminDecimal . ' / ' . $this->hostminBinary . '<br/>
			HostMax: ' . $this->hostmaxDecimal . ' / ' . $this->hostmaxBinary . '<br/>
			Hosts/Net: ' . $this->numberHosts . '<br/>';
	}
}
?>