<?php
/*
 * Project: NodeMap2.0
 * File: Network.class.php
 * Purpose: Contains a network, calculates everything like broadcast IP, netmask etc.
 */

class Network {
	public $netmaskNumber;		// Netmask number
	public $netmaskBinair;		// Binair value of netmask address
	public $netmaskDecimal;		// Decimal value of netmask address
	public $wildcardBinair;		// Binair value of wildcard address
	public $wildcardDecimal;	// Decimal value of wildcard address
	public $networkBinair;		// Binair value of network address
	public $networkDecimal;		// Decimal value of network address
	public $broadcastBinair;	// Binair value of broadcast address
	public $broadcastDecimal;	// Decimal value of broadcast address
	public $hostminBinair;		// Binair value of hostmin address
	public $hostminDecimal;		// Decimal value of hostmin address
	public $hostmaxBinair;		// Binair value of hostmax address
	public $hostmaxDecimal;		// Decimal value of hostmax address
	public $numberHosts;		// Number of possible hosts in this network

	/*
	 * Function: __construct (constructor)
	 * Description: Creating a new Network object
	 * Parameters: string $ip_address, int $ip_netmask
	 * Returns: -
	 */
	public function __construct($ip_address, $ip_netmask) {
		// Binair value of given IP address
		$binair = $this->binIP($ip_address);

		// Let's store the IP netmask number
		$this->netmaskNumber = $ip_netmask;

		// Netmask address
		$this->netmaskBinair = str_repeat('1', $ip_netmask) . str_repeat('0', 32 - $ip_netmask);
		$this->netmaskDecimal = $this->decIP($this->netmaskBinair);

		// Wildcard address
		$this->wildcardBinair = str_repeat('0', $ip_netmask) . str_repeat('1', 32 - $ip_netmask);
		$this->wildcardDecimal = $this->decIP($this->wildcardBinair);

		// Network address
		$this->networkBinair = substr($binair, 0, $ip_netmask) . str_repeat('0', 32 - $ip_netmask);
		$this->networkDecimal = $this->decIP($this->networkBinair);

		// Broadcast address
		$this->broadcastBinair = substr($binair, 0, $ip_netmask) . str_repeat('1', 32 - $ip_netmask);
		$this->broadcastDecimal = $this->decIP($this->broadcastBinair);

		// HostMin address
		$this->hostminBinair = substr($binair, 0, $ip_netmask) . str_repeat('0', 31 - $ip_netmask) . '1';
		$this->hostminDecimal = $this->decIP($this->hostminBinair);

		// HostMax address
		$this->hostmaxBinair = substr($binair, 0, $ip_netmask) . str_repeat('1', 31 - $ip_netmask) . '0';
		$this->hostmaxDecimal = $this->decIP($this->hostmaxBinair);

		// Numbers of hosts
		$this->numberHosts = bindec(substr($this->broadcastBinair, $ip_netmask)) - bindec(substr($this->networkBinair, $ip_netmask)) - 1;
	}

	function binIP($address) {
		$address_array = explode('.', $address);
		$address_binair = '';

		for ($i = 0; $i < 4; $i++) {
			// Create binair value
			$binair = decbin($address_array[$i]);

			// Trim value to the length
			$binair = trim($binair, 8);

			// Add zeros at the left to fit to $size
			$binair = str_repeat('0', 8 - strlen($binair)) . $binair;

			$address_binair .= $binair;
		}

		return $address_binair;
	}

	function decIP($binair) {
		$address_array = array();

		for ($i = 0; $i < 4; $i++) {
			$block = substr($binair, $i * 8, 8) . '<br/>';
			$address_array[] = bindec(intval($block));
		}

		return implode('.', $address_array);
	}

	function inNetwork($ip_address) {
		return false;
	}
}

$ip = '172.16.0.245';
$network = new Network($ip, 30);
echo	'
	Address: ' . $ip . ' / ' . Network::binIP($ip) . '<br/>
	Netmask: ' . $network->netmaskDecimal . ' / ' . $network->netmaskBinair . '<br/>
	Wildcard: ' . $network->wildcardDecimal . ' / ' . $network->wildcardBinair . '<br/>
	Network: ' . $network->networkDecimal . ' / ' . $network->networkBinair . '<br/>
	Broadcast: ' . $network->broadcastDecimal . ' / ' . $network->broadcastBinair . '<br/>
	HostMin: ' . $network->hostminDecimal . ' / ' . $network->hostminBinair . '<br/>
	HostMax: ' . $network->hostmaxDecimal . ' / ' . $network->hostmaxBinair . '<br/>
	Hosts/Net: ' . $network->numberHosts . '<br/>';
?>